# CVE-2026-25126: PolarLearn Vote Count Manipulation

**Research:** [Joshua van Rijswijk](https://github.com/Jvr2022)

## Description

PoC for **CVE-2026-25126**, a high-severity business logic vulnerability in the PolarLearn forum voting API.

The endpoint `POST /api/v1/forum/vote` fails to enforce strict runtime validation of the `direction` field.

Because TypeScript types are not enforced at runtime, an attacker can send arbitrary strings which are incorrectly handled downstream.

This allows manipulation of vote counts by creating so-called *ghost downvotes*.

## The Vulnerability

The API trusts the JSON body without validation:

```ts
const body: VoteRequestBody = await request.json()
const { postId, direction } = body

const result = await VoteServer(postId, direction)
```

Downstream logic treats any unexpected non-null value as a downvote, and persists invalid values.

By alternating an invalid vote with a null reset, a single user can repeatedly decrement the vote count.

## PoC Usage

### Requirements

* Node.js **18+** (native `fetch` support)
* A valid authenticated session cookie
* Access to a forum post ID to test against

### Steps

1. **Clone the repository**

```bash
git clone https://github.com/<your-username>/CVE-2026-25126-PoC.git
cd CVE-2026-25126-PoC
```

2. **Configure the PoC**

Open `poc.js` and replace:

* `POST_ID` with the target forum post ID
* `COOKIE` with your authenticated session cookie

3. **Run the PoC**

```bash
node poc.js
```

4. **Observe the result**

After each cycle, the vote count should decrease incorrectly.
Refresh the forum page to confirm the manipulated score.

## Fix

Patched by implementing strict runtime validation:

* Only accept: `"up"`, `"down"`, or `null`
* Reject all other values with HTTP 400
* Harden VoteServer against unexpected stored votes

## References

* [NVD - CVE-2026-25126](https://nvd.nist.gov/vuln/detail/CVE-2026-25126)
* [GitHub Advisory - GHSA-ghpx-5w2p-p3qp](https://github.com/polarnl/PolarLearn/security/advisories/GHSA-ghpx-5w2p-p3qp)
