# CVE-2026-25050 – Authentication Timing Attack

This repository contains a Proof of Concept demonstrating a timing attack affecting the `NativeAuthenticationStrategy.authenticate()` method.

This PoC is provided as a **technical demonstration for blog / research purposes**.

---

## Vulnerability

The authentication logic returns immediately when a user does not exist, while performing a costly password hash verification when the user exists.

```ts
const user = await this.userService.getUserByEmailAddress(ctx, data.username);
if (!user) {
    return false;
}

const passwordMatch = await this.verifyUserPassword(ctx, user.id, data.password);
```

This creates a measurable timing difference between authentication attempts for existing and non-existing users.

- Non-existing user: immediate return (~1–5 ms)
- Existing user: password hash verification (~200–400 ms with bcrypt)

By measuring response times over multiple requests, an attacker can reliably determine whether a username (email address) exists.

## Impact

- User enumeration
- Account existence disclosure
- Facilitates targeted brute-force attacks
- Facilitates phishing campaigns

## Proof of Concept

This PoC performs multiple authentication attempts per username and computes the average response time to classify accounts as:
- non-existing
- existing (password-based authentication)

## Usage

```bash
python3 exploit.py -u http://target.tld -w users.txt
```

The tool automatically appends the vulnerable endpoint:

```bash
/admin-api?languageCode=en
```

## Available Options

```bash
-u, --url              Base target URL
-w, --wordlist         Username / email wordlist
--silent               Silent mode (progress + valid users only)
--shuffle              Shuffle wordlist before testing
--delay MIN:MAX        Random delay between requests (ms)
--resume state.json    Resume a previous scan
--json results.json    Export results to JSON
```

## Notes

- Response times may vary depending on server load, network latency, and infrastructure.
- Increasing the number of attempts per username improves accuracy.
- Using `--delay` helps avoid detection and rate-limiting.
- The attack does not require valid credentials.
- This vulnerability exists due to non-constant-time authentication log

## Disclaimer

This proof of concept is provided for authorized security testing and educational purposes only.
The author assumes no responsibility for misuse.



