$TerminusDir = Split-Path -Parent $MyInvocation.MyCommand.Path

function Invoke-Terminus {
    python -m terminus @args
}

function Invoke-TerminusScan {
    python -m terminus scan @args
}

function Invoke-TerminusPassive {
    python -m terminus scan --passive-only @args
}

function Invoke-TerminusVerify {
    python -m terminus scan --verify @args
}

function Invoke-TerminusExploit {
    python -m terminus exploit @args
}

function Invoke-TerminusQuickScan {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Target
    )
    python -m terminus scan -t $Target -o terminal
}

function Invoke-TerminusVerifyScan {
    param(
        [Parameter(Mandatory=$true)]
        [string]$Target
    )
    python -m terminus scan -t $Target --verify -o terminal
}

function Invoke-TerminusBulkVerify {
    param(
        [Parameter(Mandatory=$true)]
        [string]$TargetsFile
    )
    $timestamp = Get-Date -Format "yyyyMMdd_HHmmss"
    python -m terminus scan -f $TargetsFile --verify --threads 500 -o json -w "verified_$timestamp.json"
}

function Install-TerminusCompletion {
    Write-Host "Setting up tab completion for Terminus..." -ForegroundColor Cyan
    
    $completionScript = "_TERMINUS_COMPLETE=powershell_source terminus | Out-String | Invoke-Expression"
    
    if (Test-Path $PROFILE) {
        $profileContent = Get-Content $PROFILE -Raw
        if ($profileContent -notmatch "TERMINUS_COMPLETE") {
            Add-Content -Path $PROFILE -Value "`n$completionScript"
            Write-Host "Completion added to PowerShell profile" -ForegroundColor Green
        } else {
            Write-Host "Completion already configured" -ForegroundColor Yellow
        }
    } else {
        New-Item -Path $PROFILE -ItemType File -Force | Out-Null
        Add-Content -Path $PROFILE -Value $completionScript
        Write-Host "Profile created and completion configured" -ForegroundColor Green
    }
    
    Invoke-Expression $completionScript
    Write-Host "Tab completion active in current session" -ForegroundColor Green
}

Set-Alias -Name terminus -Value Invoke-Terminus
Set-Alias -Name tscan -Value Invoke-TerminusScan
Set-Alias -Name tverify -Value Invoke-TerminusVerify
Set-Alias -Name texploit -Value Invoke-TerminusExploit

Write-Host "Terminus PowerShell helper loaded:" -ForegroundColor Cyan
Write-Host "  Invoke-TerminusScan <args>        - Run scan with custom args" -ForegroundColor Gray
Write-Host "  Invoke-TerminusPassive <target>   - Passive scan only" -ForegroundColor Gray
Write-Host "  Invoke-TerminusVerify <target>    - Active verification scan" -ForegroundColor Gray
Write-Host "  Invoke-TerminusExploit <target>   - Exploit target" -ForegroundColor Gray
Write-Host "  Invoke-TerminusQuickScan <target> - Quick single target scan" -ForegroundColor Gray
Write-Host "  Invoke-TerminusVerifyScan <target>- Quick verification scan" -ForegroundColor Gray
Write-Host "  Invoke-TerminusBulkVerify <file>  - Bulk verification with JSON output" -ForegroundColor Gray
Write-Host "  Install-TerminusCompletion        - Install tab completion" -ForegroundColor Gray
Write-Host ""
Write-Host "Aliases: terminus, tscan, tverify, texploit" -ForegroundColor Yellow
Write-Host "Example: Invoke-TerminusVerifyScan 192.168.1.100" -ForegroundColor Green
