#!/usr/bin/env python3
"""
CVE-2025-68613 - n8n Remote Exposure Checker (SAFE)
--------------------------------------------------
NO exploitation.
NO payload execution.
NO workflow creation.

Checks:
- Is the target running n8n?
- Is a version exposed?
- Is that version vulnerable to CVE-2025-68613?

Usage:
  sudo python3 CVE-2025-68613.py -u http://target
"""

import argparse
import requests
import re
import sys

requests.packages.urllib3.disable_warnings()

VULNERABLE_MIN = (0, 211, 0)
VULNERABLE_MAX_1 = (1, 120, 3)
VULNERABLE_MAX_2 = (1, 121, 0)

TIMEOUT = 8

def parse_version(text):
    match = re.search(r"(\d+)\.(\d+)\.(\d+)", text)
    if not match:
        return None
    return tuple(map(int, match.groups()))

def is_vulnerable(version):
    if version < VULNERABLE_MIN:
        return False
    if version <= VULNERABLE_MAX_1:
        return True
    if version == VULNERABLE_MAX_2:
        return True
    return False

def check_target(url):
    print(f"\n[+] Target: {url}")

    headers = {
        "User-Agent": "CVE-2025-68613-Scanner (Safe)"
    }

    paths = [
        "/",
        "/rest/settings",
        "/healthz",
        "/api/v1/health",
    ]

    for path in paths:
        try:
            r = requests.get(
                url.rstrip("/") + path,
                headers=headers,
                timeout=TIMEOUT,
                verify=False
            )

            if r.status_code >= 500:
                continue

            if "n8n" in r.text.lower() or "n8n" in str(r.headers).lower():
                print(f"[+] Possible n8n detected at {path}")

                version = parse_version(r.text)
                if version:
                    print(f"[+] Detected version: {version[0]}.{version[1]}.{version[2]}")
                    return version
                else:
                    print("[!] n8n detected but version not exposed")
                    return None

        except requests.RequestException:
            continue

    print("[-] n8n not detected or access restricted")
    return None

def main():
    parser = argparse.ArgumentParser(description="CVE-2025-68613 Safe Checker")
    parser.add_argument("-u", "--url", required=True, help="Target URL (http://host)")
    args = parser.parse_args()

    version = check_target(args.url)

    print("\n--- Result ---")

    if not version:
        print("⚠️ Unable to determine version remotely")
        print("• Target may be patched")
        print("• Version may be hidden")
        print("• Authentication may be required")
        sys.exit(1)

    if is_vulnerable(version):
        print("🚨 VULNERABLE")
        print("Target version is affected by CVE-2025-68613")
        print("Recommended: Upgrade to 1.120.4 / 1.121.1 / 1.122.0+")
        sys.exit(2)
    else:
        print("✅ NOT VULNERABLE")
        print("Detected version is not affected")
        sys.exit(0)

if __name__ == "__main__":
    main()
