#!/usr/bin/env node

/**
 * CVE-2025-66478 演示脚本
 * 展示漏洞利用、测试工具使用方法和修复步骤
 */

const { RSCExploit } = require('./exploit');
const { VulnerabilityTester } = require('./test-vulnerability');

/**
 * 模拟漏洞利用场景
 */
async function demonstrateExploit() {
    console.log('\n🔧 演示: 漏洞利用');
    console.log('-'.repeat(60));
    
    const targetUrl = 'http://localhost:3000/_rsc'; // 示例目标
    const exploit = new RSCExploit(targetUrl);
    
    try {
        // 模拟执行简单命令
        console.log('\n📝 生成并发送恶意 payload...');
        console.log('   (在真实环境中，这可能导致远程代码执行)');
        
        // 使用无害的命令进行演示
        const response = await exploit.sendExploit('echo "Hello from RCE"');
        
        console.log('🔍 响应分析:');
        console.log(`   状态码: ${response.statusCode}`);
        console.log(`   响应大小: ${response.data.length} 字节`);
        console.log(`   内容类型: ${response.headers['content-type'] || '未知'}`);
        
        // 显示部分响应内容
        if (response.data.length > 0) {
            const preview = response.data.length > 100 
                ? response.data.substring(0, 100) + '...' 
                : response.data;
            console.log(`   响应预览: ${preview}`);
        }
        
        console.log('\n⚠️  警告: 这个演示使用了无害的命令');
        console.log('   在真实攻击场景中，攻击者可以执行任意系统命令');
        
    } catch (error) {
        console.log('\n❌ 演示遇到错误 (这可能是预期的):');
        console.log(`   ${error.message}`);
        console.log('   (目标可能已关闭或不可访问)');
    }
    
    console.log('-'.repeat(60));
}

/**
 * 演示漏洞测试工具的使用
 */
async function demonstrateTesting() {
    console.log('\n🔍 演示: 漏洞测试工具');
    console.log('-'.repeat(60));
    
    const targetUrl = 'https://example.com'; // 示例目标
    const tester = new VulnerabilityTester();
    
    console.log('\n📋 测试功能概述:');
    console.log('1. 检测 Next.js 版本');
    console.log('2. 检查版本是否存在漏洞');
    console.log('3. 测试 RSC 端点');
    console.log('4. 生成漏洞报告');
    
    try {
        // 演示版本检查功能
        const testVersions = [
            '15.0.1',    // 存在漏洞
            '15.0.5',    // 已修复
            '14.2.3',    // 不受影响
            '15.1.0'     // 存在漏洞
        ];
        
        console.log('\n🔢 版本检查演示:');
        testVersions.forEach(version => {
            const isVulnerable = tester.isVersionVulnerable(version);
            const status = isVulnerable ? '🚨 存在漏洞' : '✅ 安全';
            console.log(`   ${version}: ${status}`);
        });
        
        console.log('\n📄 生成完整报告 (可能需要几秒钟):');
        console.log('   注意: 这是对 example.com 的测试，可能不会检测到 Next.js');
        
        // 生成报告 (真实环境中使用)
        // const report = await tester.generateReport(targetUrl);
        
        console.log('   测试完成 - 在真实环境中会生成详细的 JSON 报告');
        
    } catch (error) {
        console.log('\n❌ 测试工具演示遇到错误:');
        console.log(`   ${error.message}`);
    }
    
    console.log('-'.repeat(60));
}

/**
 * 演示修复步骤
 */
function demonstrateRemediation() {
    console.log('\n🛡️ 演示: 漏洞修复');
    console.log('-'.repeat(60));
    
    console.log('\n🔧 修复步骤:');
    console.log('1. 更新 Next.js 到最新版本');
    console.log('   npm install next@latest');
    console.log('   或使用修复版本:');
    console.log('   npm install next@15.0.5');
    
    console.log('\n2. 验证更新:');
    console.log('   检查 package.json 文件中的版本:');
    console.log('   grep -A 1 "next" package.json');
    
    console.log('\n3. 运行应用程序并验证:');
    console.log('   npm run build');
    console.log('   npm start');
    
    console.log('\n4. 使用测试工具验证修复:');
    console.log('   node test-vulnerability.js http://localhost:3000');
    
    console.log('\n📋 额外的安全措施:');
    console.log('• 使用 Web 应用防火墙 (WAF) 过滤恶意请求');
    console.log('• 启用内容安全策略 (CSP)');
    console.log('• 定期扫描漏洞并应用安全更新');
    console.log('• 遵循最小权限原则配置服务器');
    
    console.log('-'.repeat(60));
}

/**
 * 主函数 - 运行所有演示
 */
async function main() {
    console.log('='.repeat(60));
    console.log('CVE-2025-66478 漏洞演示');
    console.log('='.repeat(60));
    
    console.log('此演示脚本展示了:');
    console.log('1. 漏洞利用的工作原理');
    console.log('2. 测试工具的使用方法');
    console.log('3. 如何修复漏洞');
    
    console.log('\n⚠️  警告: 此脚本仅用于教育目的');
    console.log('   请只在授权的系统上使用这些工具');
    
    console.log('='.repeat(60));
    
    try {
        // 运行各个演示
        await demonstrateExploit();
        await demonstrateTesting();
        demonstrateRemediation();
        
        console.log('\n🎉 演示完成!');
        console.log('\n📚 更多信息:');
        console.log('- README.md: 漏洞详细说明');
        console.log('- USAGE.md: 工具使用指南');
        console.log('- VULNERABILITY_SUMMARY.md: 漏洞摘要');
        
    } catch (error) {
        console.error('\n❌ 演示过程中遇到错误:');
        console.error(`   ${error.message}`);
        console.error('   请检查目标配置和网络连接');
    }
    
    console.log('\n🔒 安全提示: 永远不要在生产系统上运行未经授权的测试!');
}

if (require.main === module) {
    main().catch(console.error);
}