<div align="center">

![Ivy Astronaut](assets/ivy.png)

# CVE-2025-62518: TARmageddon

</div>

This repository demonstrates a critical bug in tokio-tar and related async Rust
tar libraries where PAX extended header `size` overrides are not applied before
calculating the next header position.

## Bug Summary

**Root Cause:** When processing tar files with PAX extended headers that
override the file size, the library uses the octal size field (often zero)
instead of the PAX override for position calculations.

**Impact:** This causes the parser to jump into file content and mistake it for
tar headers, leading to extraction of wrong files.

**Known Affected Libraries:**
- `async-tar` https://github.com/dignifiedquire/async-tar
- `tokio-tar` https://github.com/vorot93/tokio-tar
- `krata-tokio-tar` https://github.com/edera-dev/tokio-tar
- `astral-tokio-tar` https://github.com/astral-sh/tokio-tar

## Quick Start

```bash
# Requirements: CMake, Rust/Cargo, C++ compiler, system tar command

cmake -S . -B build
cmake --build build --target generate_report
```

This will:
1. Build all C++ and Rust tools
2. Generate a repro case tar file
3. Run comparisons showing the bug
4. Generate detailed reports in `build/output/`

## Project Structure

```
├── disclosure/               # Security disclosure documentation tree
│   └── blast_radius/         # Record of projects depending on tokio-tar variants
├── repro_generator/          # C++ tool to generate a repro case tar file
├── tar-bug-detector/         # Rust tool comparing tar libraries  
├── tarwalk/                  # Correct C++ tar parser
│   ├── tarwalk.cpp           # Handles PAX correctly
│   └── tarwalk_bad.cpp       # Reproduces the same bug
├── CMakeLists.txt            # Build system
├── generate_report.cmake     # Report generation
└── README.md                 # This file
```

## The Bug in Detail

### Normal TAR Processing
```
Header -> Content (size from octal field) -> Next Header
```

### PAX Extended TAR Processing (Correct)
```
PAX Header (size=1024) -> File Header (octal size=0) -> Content (1024 bytes) -> Next Header
```

### PAX Extended TAR Processing (Buggy)
```
PAX Header (size=1024) -> File Header (octal size=0) -> Content (0 bytes) -> WRONG POSITION
                                                                              ↓
                                                                    Reading content as headers!
```

### Real-World Trigger

Docker `save` creates tar files with:
- Large layers (>8GB) requiring PAX extensions
- Layer content starting with filesystem tar headers (`etc/`, `usr/`)
- When the bug triggers, parsers extract filesystem content instead of image manifests

## Reproduction Files Generated

- **`pax_bug_compact.tar`** - Minimal reproduction case

## Expected Results

**Correct libraries** (GNU tar, sync `tar` crate):
```
normal.txt -> blob.bin -> marker.txt
```

**Buggy libraries** (tokio-tar):
```
normal.txt -> blob.bin -> INNER_FILE -> marker.txt
```

The appearance of `INNER_FILE` indicates the bug - the library jumped into
`blob.bin` content and mistook a fake tar header for a real entry.

## Technical Details

The fix requires applying PAX overrides before position calculations:

```rust
// Read header
let mut file_size = header.size();

// Apply PAX overrides BEFORE calculating next position
if let Some(pax_size) = pending_pax.get("size") {
    file_size = pax_size.parse().unwrap();
}

// Now calculate next header position using effective size
let next_pos = current_pos + 512 + pad_to_512(file_size);
```

## License

This reproduction code is provided for security research and responsible
disclosure purposes.

See COPYING for original source code license.
