# GitHub Security Advisory - Tarmageddon: PAX Header Desynchronization in tokio-tar

## Summary

Versions of astral-tokio-tar prior to 0.5.6 contain a boundary parsing vulnerability that allows attackers to smuggle additional archive entries by exploiting inconsistent PAX/ustar header handling. When processing archives with PAX-extended headers containing size overrides, the parser incorrectly advances stream position based on ustar header size (often zero) instead of the PAX-specified size, causing it to interpret file content as legitimate tar headers.

This vulnerability was disclosed to multiple Rust tar parsers, all derived from the original the original async-tar fork of tar-rs.

## Details

### Vulnerability Description

The vulnerability stems from inconsistent handling of PAX extended headers versus ustar headers when determining file data boundaries. Specifically:

1. **PAX header** correctly specifies the file size (e.g., `size=1048576`)
2. **ustar header** incorrectly specifies zero size (`size=000000000000`)
3. **tokio-tar** advances the stream position based on the ustar size (0 bytes)
4. **Inner content** is then interpreted as legitimate outer archive entries

### Attack Mechanism

When a TAR file contains:

- An outer entry with PAX `size=N` but ustar `size=0`
- File data that begins with valid TAR header structures
- The parser treats inner content as additional outer entries

This creates a **header/data desynchronization** where the parser's position becomes misaligned with actual file boundaries.

### Root Cause

```rust
// Vulnerable: Uses ustar size instead of PAX override
let file_size = header.size(); // Returns 0 from ustar field
let next_pos = current_pos + 512 + pad_to_512(file_size); // Advances 0 bytes

// Fixed: Apply PAX overrides before position calculation
let mut file_size = header.size();
if let Some(pax_size) = pending_pax.get("size") {
    file_size = pax_size.parse().unwrap();
}
let next_pos = current_pos + 512 + pad_to_512(file_size); // Correct advance

```

## Impact

The impact of this vulnerability depends on where astral-tokio-tar is used, and whether it is used to extract untrusted tar archives. If used to extract untrusted inputs, it may result in unexpected attacker-controlled access to the filesystem, in turn potential resulting in arbitrary code execution or credential exfiltration.

See [**GHSA-w476-p2h3-79g9**](https://github.com/astral-sh/uv/security/advisories/GHSA-w476-p2h3-79g9) for how this vulnerability affects uv, astral-tokio-tar's primary downstream user. Observe that **unlike** this advisory, uv's advisory is considered **low severity** due to overlap with intentional existing capabilities in source distributions. 

## Workarounds

Users are advised to upgrade to version 0.5.6 or newer to address this advisory.

This is no workaround other than upgrading.

## References

- [Edera blog post](https://edera.dev/stories/tarmageddon)
- [Technical Reproduction Repository](https://github.com/edera-dev/cve-tarmageddon)

## Timeline

| Date | Event |
| --- | --- |
| Aug 21, 2025 | Vulnerability discovered by Edera Security Team |
| Aug 21, 2025 | Initial analysis and PoC confirmed |
| Aug 22, 2025 | Maintainers notified (privately) |
| Aug 25, 2025 | Private patch and test suite shared |
| Oct 7, 2025 | Text freeze for GHSA |
| Oct 21, 2025 | Coordinated public disclosure and patched releases |

## Credits

- **Discovered by:** Steven Noonan (Edera) and Alex Zenla (Edera)
- **Coordinated disclosure:** Ann Wallace (Edera)
