#!/usr/bin/env python3
"""
Educational PoC for CVE-2025-61884: Oracle E-Business Suite Configurator Runtime UI Information Disclosure.
- Detects vulnerable endpoint exposure via benign HTTP request.
- For authorized testing only. Do not use on unauthorized systems.
- References: Oracle Security Alert, watchTowr Labs analysis.
"""

import argparse
import requests
from urllib.parse import urljoin
import sys

def check_vulnerability(target_url):
    """
    Sends a detection request to the vulnerable endpoint.
    Returns True if potentially vulnerable (200 OK with UI indicators).
    """
    # Vulnerable endpoint and benign parameter (safe for demo; no SSRF payload)
    endpoint = urljoin(target_url, "/OA_HTML/configurator/UiServlet")
    params = {
        'return_url': '/'  # Minimal, non-harmful redirect—tests validation bypass without targeting internals
    }
    
    headers = {
        'User-Agent': 'Mozilla/5.0 (Educational PoC Tester)',
        'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8'
    }
    
    try:
        print(f"[+] Testing endpoint: {endpoint}")
        response = requests.get(endpoint, params=params, headers=headers, timeout=10, verify=True)
        
        print(f"[+] Response Status: {response.status_code}")
        print(f"[+] Response Length: {len(response.text)} bytes")
        print(f"[+] Response Snippet: {response.text[:200]}...")  # Truncated for safety—no full dump
        
        # Basic indicators of vulnerability (non-error + UI elements)
        if response.status_code == 200 and 'Configurator' in response.text:
            print("[!] POTENTIALLY VULNERABLE: Endpoint exposed without auth. Apply Oracle patch immediately!")
            return True
        elif response.status_code in [403, 404]:
            print("[-] Not vulnerable or endpoint protected/blocked.")
            return False
        else:
            print("[?] Unexpected response—manual review recommended.")
            return False
            
    except requests.exceptions.RequestException as e:
        print(f"[-] Request failed: {e}")
        return False

def main():
    parser = argparse.ArgumentParser(description="Educational PoC for CVE-2025-61884")
    parser.add_argument("--target", required=True, help="Target EBS URL (e.g., https://example.com)")
    args = parser.parse_args()
    
    print("=== CVE-2025-61884 Educational PoC ===")
    print("WARNING: For lab/testing use only. Patch after testing.\n")
    
    vulnerable = check_vulnerability(args.target)
    
    if vulnerable:
        print("\n[INFO] Next Steps:\n- Download patch from Oracle Support.\n- Monitor logs for /UiServlet accesses.\n- Consider WAF rules blocking unauth requests to Configurator.")
    else:
        print("\n[INFO] Target appears safe, but verify with full scan.")

if __name__ == "__main__":
    main()
