# Test de la vulnérabilité SQL Injection via alias dans Django

## Définition : Qu'est-ce qu'un alias ?

Un **alias** est un nom donné à une colonne ou à un résultat d'agrégation dans une requête SQL.  
Dans Django, les alias sont utilisés avec les méthodes `annotate()` ou `alias()` pour nommer des champs calculés.

**Exemple :**

```python
from django.db.models import Count
books = Book.objects.annotate(book_count=Count('id'))
```

Ici, `book_count` est un **alias** pour le résultat de `Count('id')`.  
Cela permet de récupérer un nom lisible pour une colonne calculée.

---

## Insertion de données via le shell

Pour aller vite, on peut insérer les données via le shell :  

```bash
python manage.py shell
```

Commandes à exécuter :

```python
from myapp.models import Author, Book

a = Author.objects.create(name=" houchi pierre")
Book.objects.create(title="Les belles filles", author=a)
Book.objects.create(title="Les baux arcons", author=a)

exit()
```

---

## Contexte de la vulnérabilité

Avant Django **4.2.23**, il était possible d’injecter du SQL via des alias dynamiques fournis par l’utilisateur.  
Cette vulnérabilité est connue sous **CVE-2025-57833**.  

Elle affectait les fonctions `annotate()` et `alias()` lorsqu’un dictionnaire avec expansion (`**kwargs`) était utilisé avec des clés non filtrées provenant de l’utilisateur.

---

## Exemple de code vulnérable (`myapp/views.py`)

```python
import json
from django.db.models import Count
from django.http import JsonResponse
from .models import Author

def vulnerable_view(request):
    # On récupère l'alias depuis l'URL
    alias_param = request.GET.get("alias", "{}")
    try:
        # ⚠️ Vulnérable : évaluation directe des données utilisateurs
        alias_dict = json.loads(alias_param)
        for key, value in alias_dict.items():
            alias_dict[key] = eval(value)  # dangereux ! permet l'exécution de code arbitraire

        # Création de la queryset avec annotation dynamique
        qs = Author.objects.annotate(**alias_dict).values("name", *alias_dict.keys())

        # Retour des résultats
        return JsonResponse(list(qs), safe=False)

    except Exception as e:
        return JsonResponse({"error": str(e)})
```

### Explications :

1. `alias_param = request.GET.get("alias", "{}")`  
   Récupère la valeur du paramètre `alias` passé par l’utilisateur.

2. `alias_dict = json.loads(alias_param)`  
   Transforme la chaîne JSON en dictionnaire Python.

3. `alias_dict[key] = eval(value)`  
   ⚠️ **Très dangereux** : `eval()` exécute la chaîne comme du code Python.  
   L’utilisateur pourrait injecter du SQL destructeur.

4. `Author.objects.annotate(**alias_dict)`  
   Applique dynamiquement les annotations avec les alias fournis.

---

## Modèles (`myapp/models.py`)

```python
from django.db import models

class Author(models.Model):
    name = models.CharField(max_length=100)
    def __str__(self):
        return self.name

class Book(models.Model):
    title = models.CharField(max_length=100)
    author = models.ForeignKey(Author, on_delete=models.CASCADE)
    def __str__(self):
        return self.title
```

**Explication :**  
- `Author` et `Book` sont reliés par une relation `ForeignKey`.  
- Les annotations dynamiques sont utilisées sur `Author` pour calculer des informations sur ses livres.

---

## Exemple d’exploitation avant correction

- URL malveillante :  

```
http://127.0.0.1:8000/vuln/?alias={"evil); DROP TABLE myapp_book;--":"Count('id')"}
```

- Risque :  
  - Exécution de **toute commande SQL**, suppression de tables ou modification de données.  
  - Cela exploite directement la vulnérabilité via l’alias injecté.

---

## Correctif dans Django 4.2.23

- Django **interdit désormais certains caractères** dans les alias :  
  - Espaces  
  - Guillemets  
  - Point-virgule  
  - Commentaires SQL (`--`)

- Si l’URL contient ces caractères, Django lève une erreur :  

```
Column aliases cannot contain whitespace characters, quotation marks, semicolons, or SQL comments.
```

- Conséquence : **aucune injection SQL destructive n’est possible**.

---

## Exemple sûr

```
http://127.0.0.1:8000/vuln/?alias={"books_count":"Count('book')"}
```

- Ceci ne constitue pas une faille.  
- La faille sur les versions vulnérables (CVE-2025-57833) n’apparaît que si un utilisateur malveillant peut injecter du SQL via :  
  - des caractères interdits dans l’alias (espace, point-virgule, guillemets, commentaires SQL)  
  - ou des `eval()` sur des valeurs provenant de l’utilisateur.

---

## Bonnes pratiques pour éviter cette faille

1. Ne jamais utiliser `eval()` sur des données utilisateurs.  
2. Valider les noms d’alias pour n’accepter que des caractères sûrs.  
3. Limiter les valeurs d’annotation aux fonctions ou expressions autorisées.  
4. Maintenir Django à jour pour bénéficier des correctifs.

---

## Conclusion

- La vulnérabilité **CVE-2025-57833** permettait d’injecter du SQL via des alias dynamiques.  
- Django **4.2.23 et supérieures** corrigent cette faille.  
- La validation stricte des données utilisateurs reste indispensable.

**Auteur :** Loïc  
**Date :** 06/09/2025

