param(
    [string[]]$Frameworks = @('net6.0','net7.0','net8.0','net9.0','net10.0'),
    [string[]]$RIDs = @('win-x64','win-arm64','linux-x64','linux-arm64'),
    [string]$Configuration = 'Release',
    [bool]$PublishSingleFile = $true,
    [bool]$SelfContained = $false,
    [string]$ProjectPath = './Repro/Repro.csproj',
    [string]$OutputRoot = './artifacts',
    [switch]$PackageOutputs,
    [switch]$SkipIfPublished,
    [switch]$WhatIf
)

function Run-Publish {
    param(
        [string]$Framework,
        [string]$RID
    )

    $outDir = Join-Path -Path $OutputRoot -ChildPath "$($Framework)/$($RID)"

    $publishArgs = @(
        'publish',
        $ProjectPath,
        '-c', $Configuration,
        '-f', $Framework,
        '-r', $RID,
        '-o', $outDir,
        ('-p:PublishSingleFile=' + ($PublishSingleFile -eq $true))
    )

    if ($SelfContained) {
        $publishArgs += ('-p:SelfContained=true')
    } else {
        $publishArgs += ('-p:SelfContained=false')
    }

    $cmd = 'dotnet ' + ($publishArgs -join ' ')

    if ($WhatIf) {
        Write-Host "[WhatIf] Would run: $cmd"
        Write-Host "[WhatIf] Output directory: $outDir"
    } else {
        Write-Host "Running: $cmd"
        New-Item -ItemType Directory -Force -Path $outDir | Out-Null

        $alreadyPublished = $false
        if ($SkipIfPublished) {
            $files = Get-ChildItem -Path $outDir -File -Recurse -ErrorAction SilentlyContinue
            if ($files -and $files.Count -gt 0) {
                $alreadyPublished = $true
                Write-Host "Skipping publish for $Framework / $RID because output exists and SkipIfPublished was set."
            }
        }

        if (-not $alreadyPublished) {
            $proc = Start-Process -FilePath 'dotnet' -ArgumentList $publishArgs -NoNewWindow -Wait -PassThru
            if ($proc.ExitCode -ne 0) {
                throw "dotnet publish failed for $Framework / $RID (exit code $($proc.ExitCode))"
            }
        }
        if ($PackageOutputs) {
            Package-Output -Framework $Framework -RID $RID -OutDir $outDir
        }
    }
}

function Package-Output {
    param(
        [string]$Framework,
        [string]$RID,
        [string]$OutDir
    )

    $pkgRoot = Join-Path -Path $OutputRoot -ChildPath "packages"
    New-Item -ItemType Directory -Force -Path $pkgRoot | Out-Null

    if ($RID -like 'win-*') {
        $zipPath = Join-Path $pkgRoot "${Framework}_${RID}.zip"
        if (Test-Path $zipPath) { Remove-Item $zipPath }
        Write-Host "Packaging $OutDir -> $zipPath"
        Add-Type -AssemblyName System.IO.Compression.FileSystem
        $resolvedZipPath = Join-Path (Resolve-Path $pkgRoot).Path "${Framework}_${RID}.zip"
        [System.IO.Compression.ZipFile]::CreateFromDirectory((Resolve-Path $OutDir).Path, $resolvedZipPath)
    } else {
        $tarPath = Join-Path $pkgRoot "${Framework}_${RID}.tar.gz"
        if (Test-Path $tarPath) { Remove-Item $tarPath }
        Write-Host "Packaging $OutDir -> $tarPath"
        $tempTar = Join-Path $pkgRoot "${Framework}_${RID}.tar"
        $hasTar = (Get-Command tar -ErrorAction SilentlyContinue) -ne $null
        $hasGzip = (Get-Command gzip -ErrorAction SilentlyContinue) -ne $null
        if ($hasTar -and $hasGzip) {
            & tar -C $OutDir -cf $tempTar .
            & gzip -f $tempTar
            Move-Item -Force ($tempTar + '.gz') $tarPath
        } elseif ($hasTar -and -not $hasGzip) {
            & tar -C $OutDir -cf $tempTar .
            $zipTemp = Join-Path $pkgRoot "${Framework}_${RID}.zip"
            if (Test-Path $zipTemp) { Remove-Item $zipTemp }
            Add-Type -AssemblyName System.IO.Compression.FileSystem
            [System.IO.Compression.ZipFile]::CreateFromDirectory((Resolve-Path $OutDir).Path, $zipTemp)
            Write-Host "gzip not found; produced zip fallback at $zipTemp"
            if (Test-Path $tempTar) { Remove-Item $tempTar -Force }
        } else {
            $zipTemp = Join-Path $pkgRoot "${Framework}_${RID}.zip"
            if (Test-Path $zipTemp) { Remove-Item $zipTemp }
            Compress-Archive -Path (Join-Path $OutDir '*') -DestinationPath $zipTemp -Force
            Write-Host "Note: 'tar' not available; produced zip fallback at $pkgRoot"
        }
    }
}

foreach ($fw in $Frameworks) {
    foreach ($rid in $RIDs) {
        Run-Publish -Framework $fw -RID $rid
    }
}

Write-Host "All done."