#!/bin/bash
# CVE-2025-55182 Vulnerability Checker

echo "=== CVE-2025-55182 Vulnerability Checker ==="
echo ""
echo "Usage: Run this in your web application directory"
echo ""

if [ ! -f "package.json" ]; then
    echo "❌ No package.json found. Not a Node.js project."
    exit 1
fi

echo "📦 Checking for vulnerable packages..."
echo ""

# Check for react-server-dom-webpack
if grep -q "react-server-dom-webpack" package.json; then
    VERSION=$(grep "react-server-dom-webpack" package.json | grep -oP '\d+\.\d+\.\d+' | head -1)
    echo "⚠️  Found: react-server-dom-webpack@$VERSION"
    
    # Vulnerable versions: 19.0.0, 19.0.0-rc-*
    if [[ "$VERSION" == "19.0.0" ]] || [[ "$VERSION" =~ ^19\.0\.0-rc ]]; then
        echo "❌ VULNERABLE! This version has CVE-2025-55182"
        echo ""
        echo "Fix: npm install react-server-dom-webpack@19.1.0 (or later)"
    else
        echo "✅ Version appears safe (CVE fixed in 19.1.0+)"
    fi
else
    echo "✅ react-server-dom-webpack not found in package.json"
fi

echo ""
echo "📂 Checking installed packages..."
if [ -d "node_modules/react-server-dom-webpack" ]; then
    INSTALLED_VERSION=$(node -p "require('./node_modules/react-server-dom-webpack/package.json').version" 2>/dev/null)
    echo "   Installed version: $INSTALLED_VERSION"
    
    if [[ "$INSTALLED_VERSION" == "19.0.0" ]] || [[ "$INSTALLED_VERSION" =~ ^19\.0\.0-rc ]]; then
        echo "   ❌ VULNERABLE PACKAGE INSTALLED!"
    else
        echo "   ✅ Safe version installed"
    fi
else
    echo "   ℹ️  Package not installed"
fi

echo ""
echo "=== Server Action Usage Check ==="
echo "Searching for Server Actions in your code..."
echo ""

# Check for 'use server' directive
if grep -r "use server" --include="*.js" --include="*.jsx" --include="*.ts" --include="*.tsx" . 2>/dev/null | grep -v node_modules | head -5; then
    echo ""
    echo "⚠️  Found 'use server' directives - Server Actions are being used"
    echo "   If using vulnerable version, these endpoints are exploitable!"
else
    echo "✅ No 'use server' directives found"
fi

echo ""
echo "=== Recommendation ==="
if [[ -n "$VERSION" ]] && [[ "$VERSION" == "19.0.0" ]]; then
    echo "Action Required:"
    echo "   1. Update: npm install react-server-dom-webpack@19.1.0"
    echo "   2. Review server actions for sensitive operations"
    echo "   3. Audit server logs for suspicious activity"
else
    echo "✅ No immediate vulnerability detected"
fi
