class VirtualMemory {
  constructor(size = 0x1000) {
    this.buffer = Buffer.alloc(size);
    this.size = size;
    this.offset = 0;
  }

  allocate(bytes) {
    const aligned = (bytes + 7) & ~7; // keep 8-byte alignment for 64-bit values
    if (this.offset + aligned > this.size) {
      throw new Error(`Out of memory: requested ${aligned} bytes`);
    }
    const ptr = this.offset;
    this.offset += aligned;
    return ptr;
  }

  write64(addr, value) {
    this.#check(addr, 8);
    const bigint = BigInt.asUintN(64, BigInt(value));
    this.buffer.writeBigUInt64LE(bigint, addr);
  }

  read64(addr) {
    this.#check(addr, 8);
    return this.buffer.readBigUInt64LE(addr);
  }

  writeBytes(addr, data) {
    this.#check(addr, data.length);
    Buffer.from(data).copy(this.buffer, addr);
  }

  slice(addr, length) {
    this.#check(addr, length);
    return this.buffer.slice(addr, addr + length);
  }

  #check(addr, length) {
    if (addr < 0 || addr + length > this.size) {
      throw new RangeError(`Address ${addr} out of bounds (size ${this.size})`);
    }
  }
}

module.exports = { VirtualMemory };
