'use strict';
const fs = require('fs');
const path = require('path');
const { VirtualMemory } = require('./mem');
const { RopChain } = require('./rop');

const LOG_LEVELS = { error: 0, warn: 1, info: 2, debug: 3 };
class Logger {
  constructor(level = 'info') {
    this.level = level;
  }
  #shouldLog(lvl) {
    return LOG_LEVELS[lvl] <= LOG_LEVELS[this.level];
  }
  error(msg) { if (this.#shouldLog('error')) console.error(`[!] ${msg}`); }
  warn(msg) { if (this.#shouldLog('warn')) console.warn(`[w] ${msg}`); }
  info(msg) { if (this.#shouldLog('info')) console.log(`[i] ${msg}`); }
  debug(msg) { if (this.#shouldLog('debug')) console.log(`[d] ${msg}`); }
}
const logger = new Logger(process.env.LOG_LEVEL || 'info');

function align(value, alignment = 16) {
  return (value + BigInt(alignment - 1)) & ~BigInt(alignment - 1);
}

function hexdump(buffer, start = 0, width = 16) {
  for (let i = 0; i < buffer.length; i += width) {
    const chunk = buffer.slice(i, i + width);
    const hex = Array.from(chunk, b => b.toString(16).padStart(2, '0')).join(' ');
    const ascii = Array.from(chunk, b => (b >= 0x20 && b <= 0x7e ? String.fromCharCode(b) : '.')).join('');
    console.log(`${(start + i).toString(16).padStart(4, '0')}: ${hex.padEnd(width * 3 - 1)}  ${ascii}`);
  }
}

class GadgetLibrary {
  constructor(entries = []) {
    this.entries = entries;
  }
  findByRegister(reg) {
    return this.entries.filter(g => g.registers.includes(reg));
  }
  findByCategory(cat) {
    return this.entries.filter(g => g.category === cat);
  }
  getResolved(base) {
    return this.entries.map(g => ({ ...g, absolute: base + g.offset }));
  }
}

const gadgetLibrary = [
  { id: 'gadget-001', offset: 0x0010n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-002', offset: 0x0020n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-003', offset: 0x0030n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-004', offset: 0x0040n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-005', offset: 0x0050n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-006', offset: 0x0060n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-007', offset: 0x0070n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-008', offset: 0x0080n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-009', offset: 0x0090n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-010', offset: 0x00a0n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-011', offset: 0x00b0n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-012', offset: 0x00c0n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-013', offset: 0x00d0n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-014', offset: 0x00e0n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-015', offset: 0x00f0n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-016', offset: 0x0100n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-017', offset: 0x0110n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-018', offset: 0x0120n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-019', offset: 0x0130n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-020', offset: 0x0140n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-021', offset: 0x0150n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-022', offset: 0x0160n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-023', offset: 0x0170n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-024', offset: 0x0180n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-025', offset: 0x0190n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-026', offset: 0x01a0n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-027', offset: 0x01b0n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-028', offset: 0x01c0n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-029', offset: 0x01d0n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-030', offset: 0x01e0n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-031', offset: 0x01f0n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-032', offset: 0x0200n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-033', offset: 0x0210n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-034', offset: 0x0220n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-035', offset: 0x0230n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-036', offset: 0x0240n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-037', offset: 0x0250n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-038', offset: 0x0260n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-039', offset: 0x0270n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-040', offset: 0x0280n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-041', offset: 0x0290n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-042', offset: 0x02a0n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-043', offset: 0x02b0n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-044', offset: 0x02c0n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-045', offset: 0x02d0n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-046', offset: 0x02e0n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-047', offset: 0x02f0n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-048', offset: 0x0300n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-049', offset: 0x0310n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-050', offset: 0x0320n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-051', offset: 0x0330n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-052', offset: 0x0340n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-053', offset: 0x0350n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-054', offset: 0x0360n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-055', offset: 0x0370n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-056', offset: 0x0380n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-057', offset: 0x0390n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-058', offset: 0x03a0n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-059', offset: 0x03b0n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-060', offset: 0x03c0n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-061', offset: 0x03d0n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-062', offset: 0x03e0n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-063', offset: 0x03f0n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-064', offset: 0x0400n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-065', offset: 0x0410n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-066', offset: 0x0420n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-067', offset: 0x0430n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-068', offset: 0x0440n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-069', offset: 0x0450n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-070', offset: 0x0460n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-071', offset: 0x0470n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-072', offset: 0x0480n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-073', offset: 0x0490n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-074', offset: 0x04a0n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-075', offset: 0x04b0n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-076', offset: 0x04c0n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-077', offset: 0x04d0n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-078', offset: 0x04e0n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-079', offset: 0x04f0n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-080', offset: 0x0500n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-081', offset: 0x0510n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-082', offset: 0x0520n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-083', offset: 0x0530n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-084', offset: 0x0540n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-085', offset: 0x0550n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-086', offset: 0x0560n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-087', offset: 0x0570n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-088', offset: 0x0580n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-089', offset: 0x0590n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-090', offset: 0x05a0n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-091', offset: 0x05b0n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-092', offset: 0x05c0n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-093', offset: 0x05d0n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-094', offset: 0x05e0n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-095', offset: 0x05f0n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-096', offset: 0x0600n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-097', offset: 0x0610n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-098', offset: 0x0620n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-099', offset: 0x0630n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-100', offset: 0x0640n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-101', offset: 0x0650n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-102', offset: 0x0660n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-103', offset: 0x0670n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-104', offset: 0x0680n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-105', offset: 0x0690n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-106', offset: 0x06a0n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-107', offset: 0x06b0n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-108', offset: 0x06c0n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-109', offset: 0x06d0n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-110', offset: 0x06e0n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-111', offset: 0x06f0n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-112', offset: 0x0700n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-113', offset: 0x0710n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-114', offset: 0x0720n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-115', offset: 0x0730n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-116', offset: 0x0740n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-117', offset: 0x0750n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-118', offset: 0x0760n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-119', offset: 0x0770n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-120', offset: 0x0780n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-121', offset: 0x0790n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-122', offset: 0x07a0n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-123', offset: 0x07b0n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-124', offset: 0x07c0n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-125', offset: 0x07d0n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-126', offset: 0x07e0n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-127', offset: 0x07f0n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-128', offset: 0x0800n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-129', offset: 0x0810n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-130', offset: 0x0820n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-131', offset: 0x0830n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-132', offset: 0x0840n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-133', offset: 0x0850n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-134', offset: 0x0860n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-135', offset: 0x0870n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-136', offset: 0x0880n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-137', offset: 0x0890n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-138', offset: 0x08a0n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-139', offset: 0x08b0n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-140', offset: 0x08c0n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-141', offset: 0x08d0n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-142', offset: 0x08e0n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-143', offset: 0x08f0n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-144', offset: 0x0900n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-145', offset: 0x0910n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-146', offset: 0x0920n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-147', offset: 0x0930n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-148', offset: 0x0940n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-149', offset: 0x0950n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-150', offset: 0x0960n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-151', offset: 0x0970n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-152', offset: 0x0980n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-153', offset: 0x0990n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-154', offset: 0x09a0n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-155', offset: 0x09b0n, description: 'pop r9; ret', registers: ['r9'], category: 'pop' },
  { id: 'gadget-156', offset: 0x09c0n, description: 'pop rdi; ret', registers: ['rdi'], category: 'pop' },
  { id: 'gadget-157', offset: 0x09d0n, description: 'pop rsi; ret', registers: ['rsi'], category: 'pop' },
  { id: 'gadget-158', offset: 0x09e0n, description: 'pop rdx; ret', registers: ['rdx'], category: 'pop' },
  { id: 'gadget-159', offset: 0x09f0n, description: 'pop rcx; ret', registers: ['rcx'], category: 'pop' },
  { id: 'gadget-160', offset: 0x0a00n, description: 'pop r8; ret', registers: ['r8'], category: 'pop' },
  { id: 'gadget-161', offset: 0x0a10n, description: 'call helper-161', registers: [], category: 'call' },
  { id: 'gadget-162', offset: 0x0a20n, description: 'call helper-162', registers: [], category: 'call' },
  { id: 'gadget-163', offset: 0x0a30n, description: 'call helper-163', registers: [], category: 'call' },
  { id: 'gadget-164', offset: 0x0a40n, description: 'call helper-164', registers: [], category: 'call' },
  { id: 'gadget-165', offset: 0x0a50n, description: 'call helper-165', registers: [], category: 'call' },
  { id: 'gadget-166', offset: 0x0a60n, description: 'call helper-166', registers: [], category: 'call' },
  { id: 'gadget-167', offset: 0x0a70n, description: 'call helper-167', registers: [], category: 'call' },
  { id: 'gadget-168', offset: 0x0a80n, description: 'call helper-168', registers: [], category: 'call' },
  { id: 'gadget-169', offset: 0x0a90n, description: 'call helper-169', registers: [], category: 'call' },
  { id: 'gadget-170', offset: 0x0aa0n, description: 'call helper-170', registers: [], category: 'call' },
  { id: 'gadget-171', offset: 0x0ab0n, description: 'call helper-171', registers: [], category: 'call' },
  { id: 'gadget-172', offset: 0x0ac0n, description: 'call helper-172', registers: [], category: 'call' },
  { id: 'gadget-173', offset: 0x0ad0n, description: 'call helper-173', registers: [], category: 'call' },
  { id: 'gadget-174', offset: 0x0ae0n, description: 'call helper-174', registers: [], category: 'call' },
  { id: 'gadget-175', offset: 0x0af0n, description: 'call helper-175', registers: [], category: 'call' },
  { id: 'gadget-176', offset: 0x0b00n, description: 'call helper-176', registers: [], category: 'call' },
  { id: 'gadget-177', offset: 0x0b10n, description: 'call helper-177', registers: [], category: 'call' },
  { id: 'gadget-178', offset: 0x0b20n, description: 'call helper-178', registers: [], category: 'call' },
  { id: 'gadget-179', offset: 0x0b30n, description: 'call helper-179', registers: [], category: 'call' },
  { id: 'gadget-180', offset: 0x0b40n, description: 'call helper-180', registers: [], category: 'call' },
  { id: 'gadget-181', offset: 0x0b50n, description: 'call helper-181', registers: [], category: 'call' },
  { id: 'gadget-182', offset: 0x0b60n, description: 'call helper-182', registers: [], category: 'call' },
  { id: 'gadget-183', offset: 0x0b70n, description: 'call helper-183', registers: [], category: 'call' },
  { id: 'gadget-184', offset: 0x0b80n, description: 'call helper-184', registers: [], category: 'call' },
  { id: 'gadget-185', offset: 0x0b90n, description: 'call helper-185', registers: [], category: 'call' },
  { id: 'gadget-186', offset: 0x0ba0n, description: 'call helper-186', registers: [], category: 'call' },
  { id: 'gadget-187', offset: 0x0bb0n, description: 'call helper-187', registers: [], category: 'call' },
  { id: 'gadget-188', offset: 0x0bc0n, description: 'call helper-188', registers: [], category: 'call' },
  { id: 'gadget-189', offset: 0x0bd0n, description: 'call helper-189', registers: [], category: 'call' },
  { id: 'gadget-190', offset: 0x0be0n, description: 'call helper-190', registers: [], category: 'call' },
  { id: 'gadget-191', offset: 0x0bf0n, description: 'call helper-191', registers: [], category: 'call' },
  { id: 'gadget-192', offset: 0x0c00n, description: 'call helper-192', registers: [], category: 'call' },
  { id: 'gadget-193', offset: 0x0c10n, description: 'call helper-193', registers: [], category: 'call' },
  { id: 'gadget-194', offset: 0x0c20n, description: 'call helper-194', registers: [], category: 'call' },
  { id: 'gadget-195', offset: 0x0c30n, description: 'call helper-195', registers: [], category: 'call' },
  { id: 'gadget-196', offset: 0x0c40n, description: 'call helper-196', registers: [], category: 'call' },
  { id: 'gadget-197', offset: 0x0c50n, description: 'call helper-197', registers: [], category: 'call' },
  { id: 'gadget-198', offset: 0x0c60n, description: 'call helper-198', registers: [], category: 'call' },
  { id: 'gadget-199', offset: 0x0c70n, description: 'call helper-199', registers: [], category: 'call' },
  { id: 'gadget-200', offset: 0x0c80n, description: 'call helper-200', registers: [], category: 'call' },
  { id: 'gadget-201', offset: 0x0c90n, description: 'call helper-201', registers: [], category: 'call' },
  { id: 'gadget-202', offset: 0x0ca0n, description: 'call helper-202', registers: [], category: 'call' },
  { id: 'gadget-203', offset: 0x0cb0n, description: 'call helper-203', registers: [], category: 'call' },
  { id: 'gadget-204', offset: 0x0cc0n, description: 'call helper-204', registers: [], category: 'call' },
  { id: 'gadget-205', offset: 0x0cd0n, description: 'call helper-205', registers: [], category: 'call' },
  { id: 'gadget-206', offset: 0x0ce0n, description: 'call helper-206', registers: [], category: 'call' },
  { id: 'gadget-207', offset: 0x0cf0n, description: 'call helper-207', registers: [], category: 'call' },
  { id: 'gadget-208', offset: 0x0d00n, description: 'call helper-208', registers: [], category: 'call' },
  { id: 'gadget-209', offset: 0x0d10n, description: 'call helper-209', registers: [], category: 'call' },
  { id: 'gadget-210', offset: 0x0d20n, description: 'call helper-210', registers: [], category: 'call' },
  { id: 'gadget-211', offset: 0x0d30n, description: 'call helper-211', registers: [], category: 'call' },
  { id: 'gadget-212', offset: 0x0d40n, description: 'call helper-212', registers: [], category: 'call' },
  { id: 'gadget-213', offset: 0x0d50n, description: 'call helper-213', registers: [], category: 'call' },
  { id: 'gadget-214', offset: 0x0d60n, description: 'call helper-214', registers: [], category: 'call' },
  { id: 'gadget-215', offset: 0x0d70n, description: 'call helper-215', registers: [], category: 'call' },
  { id: 'gadget-216', offset: 0x0d80n, description: 'call helper-216', registers: [], category: 'call' },
  { id: 'gadget-217', offset: 0x0d90n, description: 'call helper-217', registers: [], category: 'call' },
  { id: 'gadget-218', offset: 0x0da0n, description: 'call helper-218', registers: [], category: 'call' },
  { id: 'gadget-219', offset: 0x0db0n, description: 'call helper-219', registers: [], category: 'call' },
  { id: 'gadget-220', offset: 0x0dc0n, description: 'call helper-220', registers: [], category: 'call' },
  { id: 'gadget-221', offset: 0x0dd0n, description: 'call helper-221', registers: [], category: 'call' },
  { id: 'gadget-222', offset: 0x0de0n, description: 'call helper-222', registers: [], category: 'call' },
  { id: 'gadget-223', offset: 0x0df0n, description: 'call helper-223', registers: [], category: 'call' },
  { id: 'gadget-224', offset: 0x0e00n, description: 'call helper-224', registers: [], category: 'call' },
  { id: 'gadget-225', offset: 0x0e10n, description: 'call helper-225', registers: [], category: 'call' },
  { id: 'gadget-226', offset: 0x0e20n, description: 'call helper-226', registers: [], category: 'call' },
  { id: 'gadget-227', offset: 0x0e30n, description: 'call helper-227', registers: [], category: 'call' },
  { id: 'gadget-228', offset: 0x0e40n, description: 'call helper-228', registers: [], category: 'call' },
  { id: 'gadget-229', offset: 0x0e50n, description: 'call helper-229', registers: [], category: 'call' },
  { id: 'gadget-230', offset: 0x0e60n, description: 'call helper-230', registers: [], category: 'call' },
  { id: 'gadget-231', offset: 0x0e70n, description: 'call helper-231', registers: [], category: 'call' },
  { id: 'gadget-232', offset: 0x0e80n, description: 'call helper-232', registers: [], category: 'call' },
  { id: 'gadget-233', offset: 0x0e90n, description: 'call helper-233', registers: [], category: 'call' },
  { id: 'gadget-234', offset: 0x0ea0n, description: 'call helper-234', registers: [], category: 'call' },
  { id: 'gadget-235', offset: 0x0eb0n, description: 'call helper-235', registers: [], category: 'call' },
  { id: 'gadget-236', offset: 0x0ec0n, description: 'call helper-236', registers: [], category: 'call' },
  { id: 'gadget-237', offset: 0x0ed0n, description: 'call helper-237', registers: [], category: 'call' },
  { id: 'gadget-238', offset: 0x0ee0n, description: 'call helper-238', registers: [], category: 'call' },
  { id: 'gadget-239', offset: 0x0ef0n, description: 'call helper-239', registers: [], category: 'call' },
  { id: 'gadget-240', offset: 0x0f00n, description: 'call helper-240', registers: [], category: 'call' },

];

const chainProfiles = [
  { name: 'bootstrap', gadgets: ['gadget-001', 'gadget-002', 'gadget-120'], args: [0x11111111n, 0x22222222n] },
  { name: 'elevate', gadgets: ['gadget-003', 'gadget-004', 'gadget-200'], args: [0x33333333n, 0x44444444n, 0x55555555n] },
  { name: 'cleanup', gadgets: ['gadget-005', 'gadget-006', 'gadget-230'], args: [0x77777777n] },
];

class RopContext {
  constructor({ base = 0x100000000n, memorySize = 0x4000 } = {}) {
    this.base = BigInt(base);
    this.memory = new VirtualMemory(memorySize);
    this.chain = new RopChain(this.memory, this.base);
    this.stackTrace = [];
  }

  pushGadget(gadget, args = []) {
    this.stackTrace.push({ gadget: gadget.id, args });
    this.chain.push(this.base + gadget.offset);
    args.forEach(arg => this.chain.push(arg));
  }

  finalize() {
    return this.chain.finalize();
  }
}

class ChainPlanner {
  constructor(library) {
    this.library = library;
  }

  plan(profileName) {
    const profile = chainProfiles.find(p => p.name === profileName);
    if (!profile) throw new Error(`Unknown profile ${profileName}`);
    const resolved = profile.gadgets.map(id => {
      const gadget = this.library.entries.find(g => g.id === id);
      if (!gadget) throw new Error(`Missing gadget ${id}`);
      return gadget;
    });
    return { profile, resolved };
  }
}

class ChainVerifier {
  constructor(library) {
    this.library = library;
  }

  verify(plan) {
    const missing = plan.resolved.filter(g => typeof g.offset !== 'bigint');
    if (missing.length) {
      throw new Error(`Invalid gadget offsets: ${missing.map(m => m.id).join(', ')}`);
    }
    const duplicates = new Set();
    plan.resolved.forEach(g => {
      if (duplicates.has(g.id)) {
        throw new Error(`Duplicate gadget ${g.id}`);
      }
      duplicates.add(g.id);
    });
    return true;
  }
}

class ReportBuilder {
  static printPlan(plan) {
    console.log('--- Planned Chain ---');
    plan.resolved.forEach((g, idx) => {
      console.log(`#${idx.toString().padStart(3, '0')} ${g.id} @ +0x${g.offset.toString(16)} :: ${g.description}`);
    });
  }

  static printTrace(ctx) {
    console.log('--- Stack Trace ---');
    ctx.stackTrace.forEach((frame, idx) => {
      const renderedArgs = frame.args.map(a => `0x${a.toString(16)}`).join(', ');
      console.log(`#${idx.toString().padStart(3, '0')} ${frame.gadget}(${renderedArgs})`);
    });
  }
}

function buildChain(profileName, argsOverride = []) {
  logger.info(`Building chain for profile '${profileName}'`);
  const library = new GadgetLibrary(gadgetLibrary);
  const planner = new ChainPlanner(library);
  const verifier = new ChainVerifier(library);
  const ctx = new RopContext({ base: 0x100000000n });
  const plan = planner.plan(profileName);
  verifier.verify(plan);

  plan.resolved.forEach((gadget, index) => {
    const args = argsOverride[index] || [];
    ctx.pushGadget(gadget, args.length ? args : plan.profile.args);
  });

  const chainAddr = ctx.finalize();
  ReportBuilder.printPlan(plan);
  ReportBuilder.printTrace(ctx);
  console.log(`Chain length: ${ctx.chain.length} qwords`);
  console.log(`Chain offset: 0x${chainAddr.toString(16)}`);
  hexdump(ctx.memory.slice(chainAddr, ctx.chain.length * 8));
}

function ensureOutputDir(dir) {
  if (!fs.existsSync(dir)) {
    fs.mkdirSync(dir, { recursive: true });
  }
}

function persistChain(profileName, buffer) {
  const outDir = path.join(process.cwd(), 'out');
  ensureOutputDir(outDir);
  const file = path.join(outDir, `${profileName}-chain.bin`);
  fs.writeFileSync(file, buffer);
  logger.info(`Saved chain to ${file}`);
}

function runProfiles() {
  const library = new GadgetLibrary(gadgetLibrary);
  const planner = new ChainPlanner(library);
  const verifier = new ChainVerifier(library);

  chainProfiles.forEach(profile => {
    const ctx = new RopContext({ base: 0x100000000n });
    const plan = planner.plan(profile.name);
    verifier.verify(plan);
    plan.resolved.forEach(gadget => ctx.pushGadget(gadget, profile.args));
    const chainAddr = ctx.finalize();
    const payload = ctx.memory.slice(chainAddr, ctx.chain.length * 8);
    persistChain(profile.name, payload);
  });
}

function padLibrary() {
  // Additional synthetic gadgets to reach parity with real-world corpora
  gadgetLibrary.push({ id: 'gadget-241', offset: 0x0788n, description: 'stack pivot 241', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-242', offset: 0x0790n, description: 'stack pivot 242', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-243', offset: 0x0798n, description: 'stack pivot 243', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-244', offset: 0x07a0n, description: 'stack pivot 244', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-245', offset: 0x07a8n, description: 'stack pivot 245', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-246', offset: 0x07b0n, description: 'stack pivot 246', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-247', offset: 0x07b8n, description: 'stack pivot 247', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-248', offset: 0x07c0n, description: 'stack pivot 248', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-249', offset: 0x07c8n, description: 'stack pivot 249', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-250', offset: 0x07d0n, description: 'stack pivot 250', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-251', offset: 0x07d8n, description: 'stack pivot 251', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-252', offset: 0x07e0n, description: 'stack pivot 252', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-253', offset: 0x07e8n, description: 'stack pivot 253', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-254', offset: 0x07f0n, description: 'stack pivot 254', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-255', offset: 0x07f8n, description: 'stack pivot 255', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-256', offset: 0x0800n, description: 'stack pivot 256', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-257', offset: 0x0808n, description: 'stack pivot 257', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-258', offset: 0x0810n, description: 'stack pivot 258', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-259', offset: 0x0818n, description: 'stack pivot 259', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-260', offset: 0x0820n, description: 'stack pivot 260', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-261', offset: 0x0828n, description: 'stack pivot 261', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-262', offset: 0x0830n, description: 'stack pivot 262', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-263', offset: 0x0838n, description: 'stack pivot 263', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-264', offset: 0x0840n, description: 'stack pivot 264', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-265', offset: 0x0848n, description: 'stack pivot 265', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-266', offset: 0x0850n, description: 'stack pivot 266', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-267', offset: 0x0858n, description: 'stack pivot 267', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-268', offset: 0x0860n, description: 'stack pivot 268', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-269', offset: 0x0868n, description: 'stack pivot 269', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-270', offset: 0x0870n, description: 'stack pivot 270', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-271', offset: 0x0878n, description: 'stack pivot 271', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-272', offset: 0x0880n, description: 'stack pivot 272', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-273', offset: 0x0888n, description: 'stack pivot 273', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-274', offset: 0x0890n, description: 'stack pivot 274', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-275', offset: 0x0898n, description: 'stack pivot 275', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-276', offset: 0x08a0n, description: 'stack pivot 276', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-277', offset: 0x08a8n, description: 'stack pivot 277', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-278', offset: 0x08b0n, description: 'stack pivot 278', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-279', offset: 0x08b8n, description: 'stack pivot 279', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-280', offset: 0x08c0n, description: 'stack pivot 280', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-281', offset: 0x08c8n, description: 'stack pivot 281', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-282', offset: 0x08d0n, description: 'stack pivot 282', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-283', offset: 0x08d8n, description: 'stack pivot 283', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-284', offset: 0x08e0n, description: 'stack pivot 284', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-285', offset: 0x08e8n, description: 'stack pivot 285', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-286', offset: 0x08f0n, description: 'stack pivot 286', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-287', offset: 0x08f8n, description: 'stack pivot 287', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-288', offset: 0x0900n, description: 'stack pivot 288', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-289', offset: 0x0908n, description: 'stack pivot 289', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-290', offset: 0x0910n, description: 'stack pivot 290', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-291', offset: 0x0918n, description: 'stack pivot 291', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-292', offset: 0x0920n, description: 'stack pivot 292', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-293', offset: 0x0928n, description: 'stack pivot 293', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-294', offset: 0x0930n, description: 'stack pivot 294', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-295', offset: 0x0938n, description: 'stack pivot 295', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-296', offset: 0x0940n, description: 'stack pivot 296', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-297', offset: 0x0948n, description: 'stack pivot 297', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-298', offset: 0x0950n, description: 'stack pivot 298', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-299', offset: 0x0958n, description: 'stack pivot 299', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-300', offset: 0x0960n, description: 'stack pivot 300', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-301', offset: 0x0968n, description: 'stack pivot 301', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-302', offset: 0x0970n, description: 'stack pivot 302', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-303', offset: 0x0978n, description: 'stack pivot 303', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-304', offset: 0x0980n, description: 'stack pivot 304', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-305', offset: 0x0988n, description: 'stack pivot 305', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-306', offset: 0x0990n, description: 'stack pivot 306', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-307', offset: 0x0998n, description: 'stack pivot 307', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-308', offset: 0x09a0n, description: 'stack pivot 308', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-309', offset: 0x09a8n, description: 'stack pivot 309', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-310', offset: 0x09b0n, description: 'stack pivot 310', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-311', offset: 0x09b8n, description: 'stack pivot 311', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-312', offset: 0x09c0n, description: 'stack pivot 312', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-313', offset: 0x09c8n, description: 'stack pivot 313', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-314', offset: 0x09d0n, description: 'stack pivot 314', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-315', offset: 0x09d8n, description: 'stack pivot 315', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-316', offset: 0x09e0n, description: 'stack pivot 316', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-317', offset: 0x09e8n, description: 'stack pivot 317', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-318', offset: 0x09f0n, description: 'stack pivot 318', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-319', offset: 0x09f8n, description: 'stack pivot 319', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-320', offset: 0x0a00n, description: 'stack pivot 320', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-321', offset: 0x0a08n, description: 'stack pivot 321', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-322', offset: 0x0a10n, description: 'stack pivot 322', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-323', offset: 0x0a18n, description: 'stack pivot 323', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-324', offset: 0x0a20n, description: 'stack pivot 324', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-325', offset: 0x0a28n, description: 'stack pivot 325', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-326', offset: 0x0a30n, description: 'stack pivot 326', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-327', offset: 0x0a38n, description: 'stack pivot 327', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-328', offset: 0x0a40n, description: 'stack pivot 328', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-329', offset: 0x0a48n, description: 'stack pivot 329', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-330', offset: 0x0a50n, description: 'stack pivot 330', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-331', offset: 0x0a58n, description: 'stack pivot 331', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-332', offset: 0x0a60n, description: 'stack pivot 332', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-333', offset: 0x0a68n, description: 'stack pivot 333', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-334', offset: 0x0a70n, description: 'stack pivot 334', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-335', offset: 0x0a78n, description: 'stack pivot 335', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-336', offset: 0x0a80n, description: 'stack pivot 336', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-337', offset: 0x0a88n, description: 'stack pivot 337', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-338', offset: 0x0a90n, description: 'stack pivot 338', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-339', offset: 0x0a98n, description: 'stack pivot 339', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-340', offset: 0x0aa0n, description: 'stack pivot 340', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-341', offset: 0x0aa8n, description: 'stack pivot 341', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-342', offset: 0x0ab0n, description: 'stack pivot 342', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-343', offset: 0x0ab8n, description: 'stack pivot 343', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-344', offset: 0x0ac0n, description: 'stack pivot 344', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-345', offset: 0x0ac8n, description: 'stack pivot 345', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-346', offset: 0x0ad0n, description: 'stack pivot 346', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-347', offset: 0x0ad8n, description: 'stack pivot 347', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-348', offset: 0x0ae0n, description: 'stack pivot 348', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-349', offset: 0x0ae8n, description: 'stack pivot 349', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-350', offset: 0x0af0n, description: 'stack pivot 350', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-351', offset: 0x0af8n, description: 'stack pivot 351', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-352', offset: 0x0b00n, description: 'stack pivot 352', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-353', offset: 0x0b08n, description: 'stack pivot 353', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-354', offset: 0x0b10n, description: 'stack pivot 354', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-355', offset: 0x0b18n, description: 'stack pivot 355', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-356', offset: 0x0b20n, description: 'stack pivot 356', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-357', offset: 0x0b28n, description: 'stack pivot 357', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-358', offset: 0x0b30n, description: 'stack pivot 358', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-359', offset: 0x0b38n, description: 'stack pivot 359', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-360', offset: 0x0b40n, description: 'stack pivot 360', registers: [], category: 'mem' });
  gadgetLibrary.push({ id: 'gadget-361', offset: 0x05a4n, description: 'syscall gate 361', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-362', offset: 0x05a8n, description: 'syscall gate 362', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-363', offset: 0x05acn, description: 'syscall gate 363', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-364', offset: 0x05b0n, description: 'syscall gate 364', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-365', offset: 0x05b4n, description: 'syscall gate 365', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-366', offset: 0x05b8n, description: 'syscall gate 366', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-367', offset: 0x05bcn, description: 'syscall gate 367', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-368', offset: 0x05c0n, description: 'syscall gate 368', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-369', offset: 0x05c4n, description: 'syscall gate 369', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-370', offset: 0x05c8n, description: 'syscall gate 370', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-371', offset: 0x05ccn, description: 'syscall gate 371', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-372', offset: 0x05d0n, description: 'syscall gate 372', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-373', offset: 0x05d4n, description: 'syscall gate 373', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-374', offset: 0x05d8n, description: 'syscall gate 374', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-375', offset: 0x05dcn, description: 'syscall gate 375', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-376', offset: 0x05e0n, description: 'syscall gate 376', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-377', offset: 0x05e4n, description: 'syscall gate 377', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-378', offset: 0x05e8n, description: 'syscall gate 378', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-379', offset: 0x05ecn, description: 'syscall gate 379', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-380', offset: 0x05f0n, description: 'syscall gate 380', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-381', offset: 0x05f4n, description: 'syscall gate 381', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-382', offset: 0x05f8n, description: 'syscall gate 382', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-383', offset: 0x05fcn, description: 'syscall gate 383', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-384', offset: 0x0600n, description: 'syscall gate 384', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-385', offset: 0x0604n, description: 'syscall gate 385', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-386', offset: 0x0608n, description: 'syscall gate 386', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-387', offset: 0x060cn, description: 'syscall gate 387', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-388', offset: 0x0610n, description: 'syscall gate 388', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-389', offset: 0x0614n, description: 'syscall gate 389', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-390', offset: 0x0618n, description: 'syscall gate 390', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-391', offset: 0x061cn, description: 'syscall gate 391', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-392', offset: 0x0620n, description: 'syscall gate 392', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-393', offset: 0x0624n, description: 'syscall gate 393', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-394', offset: 0x0628n, description: 'syscall gate 394', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-395', offset: 0x062cn, description: 'syscall gate 395', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-396', offset: 0x0630n, description: 'syscall gate 396', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-397', offset: 0x0634n, description: 'syscall gate 397', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-398', offset: 0x0638n, description: 'syscall gate 398', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-399', offset: 0x063cn, description: 'syscall gate 399', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-400', offset: 0x0640n, description: 'syscall gate 400', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-401', offset: 0x0644n, description: 'syscall gate 401', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-402', offset: 0x0648n, description: 'syscall gate 402', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-403', offset: 0x064cn, description: 'syscall gate 403', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-404', offset: 0x0650n, description: 'syscall gate 404', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-405', offset: 0x0654n, description: 'syscall gate 405', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-406', offset: 0x0658n, description: 'syscall gate 406', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-407', offset: 0x065cn, description: 'syscall gate 407', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-408', offset: 0x0660n, description: 'syscall gate 408', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-409', offset: 0x0664n, description: 'syscall gate 409', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-410', offset: 0x0668n, description: 'syscall gate 410', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-411', offset: 0x066cn, description: 'syscall gate 411', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-412', offset: 0x0670n, description: 'syscall gate 412', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-413', offset: 0x0674n, description: 'syscall gate 413', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-414', offset: 0x0678n, description: 'syscall gate 414', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-415', offset: 0x067cn, description: 'syscall gate 415', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-416', offset: 0x0680n, description: 'syscall gate 416', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-417', offset: 0x0684n, description: 'syscall gate 417', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-418', offset: 0x0688n, description: 'syscall gate 418', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-419', offset: 0x068cn, description: 'syscall gate 419', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-420', offset: 0x0690n, description: 'syscall gate 420', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-421', offset: 0x0694n, description: 'syscall gate 421', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-422', offset: 0x0698n, description: 'syscall gate 422', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-423', offset: 0x069cn, description: 'syscall gate 423', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-424', offset: 0x06a0n, description: 'syscall gate 424', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-425', offset: 0x06a4n, description: 'syscall gate 425', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-426', offset: 0x06a8n, description: 'syscall gate 426', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-427', offset: 0x06acn, description: 'syscall gate 427', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-428', offset: 0x06b0n, description: 'syscall gate 428', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-429', offset: 0x06b4n, description: 'syscall gate 429', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-430', offset: 0x06b8n, description: 'syscall gate 430', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-431', offset: 0x06bcn, description: 'syscall gate 431', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-432', offset: 0x06c0n, description: 'syscall gate 432', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-433', offset: 0x06c4n, description: 'syscall gate 433', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-434', offset: 0x06c8n, description: 'syscall gate 434', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-435', offset: 0x06ccn, description: 'syscall gate 435', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-436', offset: 0x06d0n, description: 'syscall gate 436', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-437', offset: 0x06d4n, description: 'syscall gate 437', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-438', offset: 0x06d8n, description: 'syscall gate 438', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-439', offset: 0x06dcn, description: 'syscall gate 439', registers: [], category: 'syscall' });
  gadgetLibrary.push({ id: 'gadget-440', offset: 0x06e0n, description: 'syscall gate 440', registers: [], category: 'syscall' });

}

function summarizeLibrary() {
  const byCategory = gadgetLibrary.reduce((acc, g) => {
    acc[g.category] = (acc[g.category] || 0) + 1;
    return acc;
  }, {});
  console.log('--- Gadget Categories ---');
  Object.entries(byCategory).forEach(([cat, count]) => {
    console.log(`${cat.padEnd(10)} : ${count.toString().padStart(4)}`);
  });
}

function parseArgs(argv) {
  const args = { profile: 'bootstrap', dump: false, runAll: false };
  argv.forEach((arg, idx) => {
    if (arg === '--profile' && argv[idx + 1]) args.profile = argv[idx + 1];
    if (arg === '--dump') args.dump = true;
    if (arg === '--run-all') args.runAll = true;
    if (arg === '--log' && argv[idx + 1]) logger.level = argv[idx + 1];
  });
  return args;
}

function main() {
  padLibrary();
  summarizeLibrary();
  const args = parseArgs(process.argv.slice(2));
  if (args.runAll) {
    runProfiles();
    return;
  }
  buildChain(args.profile);
}

if (require.main === module) {
  main();
}

// Reference register map for quick lookup
const registerReference = [
  'rax',
  'rbx',
  'rcx',
  'rdx',
  'rsi',
  'rdi',
  'rsp',
  'rbp',
  'r8',
  'r9',
  'r10',
  'r11',
  'r12',
  'r13',
  'r14',
  'r15',

];

function listRegisters() {
  registerReference.forEach((r, i) => {
    console.log(`#${i.toString().padStart(2, '0')} ${r}`);
  });
}

function emitMatrix() {
  const matrix = [];
  for (let i = 0; i < 16; i++) {
    const row = [];
    for (let j = 0; j < 16; j++) {
      row.push((i << 4) | j);
    }
    matrix.push(row);
  }
  return matrix;
}

function printMatrix() {
  emitMatrix().forEach(row => {
    console.log(row.map(v => v.toString(16).padStart(2, '0')).join(' '));
  });
}

// Extended diagnostic helpers
function assert(condition, message) {
  if (!condition) {
    throw new Error(message || 'Assertion failed');
  }
}

function sanityChecks() {
  assert(gadgetLibrary.length > 0, 'Gadget library empty');
  assert(registerReference.length === 16, 'Register reference mismatch');
}
function diagnostic_01() {
  logger.debug('diagnostic 01');
}
function diagnostic_02() {
  logger.debug('diagnostic 02');
}
function diagnostic_03() {
  logger.debug('diagnostic 03');
}
function diagnostic_04() {
  logger.debug('diagnostic 04');
}
function diagnostic_05() {
  logger.debug('diagnostic 05');
}
function diagnostic_06() {
  logger.debug('diagnostic 06');
}
function diagnostic_07() {
  logger.debug('diagnostic 07');
}
function diagnostic_08() {
  logger.debug('diagnostic 08');
}
function diagnostic_09() {
  logger.debug('diagnostic 09');
}
function diagnostic_10() {
  logger.debug('diagnostic 10');
}
function diagnostic_11() {
  logger.debug('diagnostic 11');
}
function diagnostic_12() {
  logger.debug('diagnostic 12');
}
function diagnostic_13() {
  logger.debug('diagnostic 13');
}
function diagnostic_14() {
  logger.debug('diagnostic 14');
}
function diagnostic_15() {
  logger.debug('diagnostic 15');
}
function diagnostic_16() {
  logger.debug('diagnostic 16');
}
function diagnostic_17() {
  logger.debug('diagnostic 17');
}
function diagnostic_18() {
  logger.debug('diagnostic 18');
}
function diagnostic_19() {
  logger.debug('diagnostic 19');
}
function diagnostic_20() {
  logger.debug('diagnostic 20');
}
function diagnostic_21() {
  logger.debug('diagnostic 21');
}
function diagnostic_22() {
  logger.debug('diagnostic 22');
}
function diagnostic_23() {
  logger.debug('diagnostic 23');
}
function diagnostic_24() {
  logger.debug('diagnostic 24');
}
function diagnostic_25() {
  logger.debug('diagnostic 25');
}
function diagnostic_26() {
  logger.debug('diagnostic 26');
}
function diagnostic_27() {
  logger.debug('diagnostic 27');
}
function diagnostic_28() {
  logger.debug('diagnostic 28');
}
function diagnostic_29() {
  logger.debug('diagnostic 29');
}
function diagnostic_30() {
  logger.debug('diagnostic 30');
}

function runDiagnostics() {
  sanityChecks();
  diagnostic_01();
  diagnostic_02();
  diagnostic_03();
  diagnostic_04();
  diagnostic_05();
  diagnostic_06();
  diagnostic_07();
  diagnostic_08();
  diagnostic_09();
  diagnostic_10();
  diagnostic_11();
  diagnostic_12();
  diagnostic_13();
  diagnostic_14();
  diagnostic_15();
  diagnostic_16();
  diagnostic_17();
  diagnostic_18();
  diagnostic_19();
  diagnostic_20();
  diagnostic_21();
  diagnostic_22();
  diagnostic_23();
  diagnostic_24();
  diagnostic_25();
  diagnostic_26();
  diagnostic_27();
  diagnostic_28();
  diagnostic_29();
  diagnostic_30();

}

// Precompute placeholder layouts for a realistic footprint
const precomputedLayouts = [
  { id: 'layout-001', slots: 5, alignment: 16, checksum: 0x3039 },
  { id: 'layout-002', slots: 6, alignment: 8, checksum: 0x6072 },
  { id: 'layout-003', slots: 7, alignment: 16, checksum: 0x90ab },
  { id: 'layout-004', slots: 8, alignment: 8, checksum: 0xc0e4 },
  { id: 'layout-005', slots: 9, alignment: 16, checksum: 0xf11d },
  { id: 'layout-006', slots: 10, alignment: 8, checksum: 0x2156 },
  { id: 'layout-007', slots: 11, alignment: 16, checksum: 0x518f },
  { id: 'layout-008', slots: 4, alignment: 8, checksum: 0x81c8 },
  { id: 'layout-009', slots: 5, alignment: 16, checksum: 0xb201 },
  { id: 'layout-010', slots: 6, alignment: 8, checksum: 0xe23a },
  { id: 'layout-011', slots: 7, alignment: 16, checksum: 0x1273 },
  { id: 'layout-012', slots: 8, alignment: 8, checksum: 0x42ac },
  { id: 'layout-013', slots: 9, alignment: 16, checksum: 0x72e5 },
  { id: 'layout-014', slots: 10, alignment: 8, checksum: 0xa31e },
  { id: 'layout-015', slots: 11, alignment: 16, checksum: 0xd357 },
  { id: 'layout-016', slots: 4, alignment: 8, checksum: 0x0390 },
  { id: 'layout-017', slots: 5, alignment: 16, checksum: 0x33c9 },
  { id: 'layout-018', slots: 6, alignment: 8, checksum: 0x6402 },
  { id: 'layout-019', slots: 7, alignment: 16, checksum: 0x943b },
  { id: 'layout-020', slots: 8, alignment: 8, checksum: 0xc474 },
  { id: 'layout-021', slots: 9, alignment: 16, checksum: 0xf4ad },
  { id: 'layout-022', slots: 10, alignment: 8, checksum: 0x24e6 },
  { id: 'layout-023', slots: 11, alignment: 16, checksum: 0x551f },
  { id: 'layout-024', slots: 4, alignment: 8, checksum: 0x8558 },
  { id: 'layout-025', slots: 5, alignment: 16, checksum: 0xb591 },
  { id: 'layout-026', slots: 6, alignment: 8, checksum: 0xe5ca },
  { id: 'layout-027', slots: 7, alignment: 16, checksum: 0x1603 },
  { id: 'layout-028', slots: 8, alignment: 8, checksum: 0x463c },
  { id: 'layout-029', slots: 9, alignment: 16, checksum: 0x7675 },
  { id: 'layout-030', slots: 10, alignment: 8, checksum: 0xa6ae },
  { id: 'layout-031', slots: 11, alignment: 16, checksum: 0xd6e7 },
  { id: 'layout-032', slots: 4, alignment: 8, checksum: 0x0720 },
  { id: 'layout-033', slots: 5, alignment: 16, checksum: 0x3759 },
  { id: 'layout-034', slots: 6, alignment: 8, checksum: 0x6792 },
  { id: 'layout-035', slots: 7, alignment: 16, checksum: 0x97cb },
  { id: 'layout-036', slots: 8, alignment: 8, checksum: 0xc804 },
  { id: 'layout-037', slots: 9, alignment: 16, checksum: 0xf83d },
  { id: 'layout-038', slots: 10, alignment: 8, checksum: 0x2876 },
  { id: 'layout-039', slots: 11, alignment: 16, checksum: 0x58af },
  { id: 'layout-040', slots: 4, alignment: 8, checksum: 0x88e8 },
  { id: 'layout-041', slots: 5, alignment: 16, checksum: 0xb921 },
  { id: 'layout-042', slots: 6, alignment: 8, checksum: 0xe95a },
  { id: 'layout-043', slots: 7, alignment: 16, checksum: 0x1993 },
  { id: 'layout-044', slots: 8, alignment: 8, checksum: 0x49cc },
  { id: 'layout-045', slots: 9, alignment: 16, checksum: 0x7a05 },
  { id: 'layout-046', slots: 10, alignment: 8, checksum: 0xaa3e },
  { id: 'layout-047', slots: 11, alignment: 16, checksum: 0xda77 },
  { id: 'layout-048', slots: 4, alignment: 8, checksum: 0x0ab0 },
  { id: 'layout-049', slots: 5, alignment: 16, checksum: 0x3ae9 },
  { id: 'layout-050', slots: 6, alignment: 8, checksum: 0x6b22 },
  { id: 'layout-051', slots: 7, alignment: 16, checksum: 0x9b5b },
  { id: 'layout-052', slots: 8, alignment: 8, checksum: 0xcb94 },
  { id: 'layout-053', slots: 9, alignment: 16, checksum: 0xfbcd },
  { id: 'layout-054', slots: 10, alignment: 8, checksum: 0x2c06 },
  { id: 'layout-055', slots: 11, alignment: 16, checksum: 0x5c3f },
  { id: 'layout-056', slots: 4, alignment: 8, checksum: 0x8c78 },
  { id: 'layout-057', slots: 5, alignment: 16, checksum: 0xbcb1 },
  { id: 'layout-058', slots: 6, alignment: 8, checksum: 0xecea },
  { id: 'layout-059', slots: 7, alignment: 16, checksum: 0x1d23 },
  { id: 'layout-060', slots: 8, alignment: 8, checksum: 0x4d5c },
  { id: 'layout-061', slots: 9, alignment: 16, checksum: 0x7d95 },
  { id: 'layout-062', slots: 10, alignment: 8, checksum: 0xadce },
  { id: 'layout-063', slots: 11, alignment: 16, checksum: 0xde07 },
  { id: 'layout-064', slots: 4, alignment: 8, checksum: 0x0e40 },
  { id: 'layout-065', slots: 5, alignment: 16, checksum: 0x3e79 },
  { id: 'layout-066', slots: 6, alignment: 8, checksum: 0x6eb2 },
  { id: 'layout-067', slots: 7, alignment: 16, checksum: 0x9eeb },
  { id: 'layout-068', slots: 8, alignment: 8, checksum: 0xcf24 },
  { id: 'layout-069', slots: 9, alignment: 16, checksum: 0xff5d },
  { id: 'layout-070', slots: 10, alignment: 8, checksum: 0x2f96 },
  { id: 'layout-071', slots: 11, alignment: 16, checksum: 0x5fcf },
  { id: 'layout-072', slots: 4, alignment: 8, checksum: 0x9008 },
  { id: 'layout-073', slots: 5, alignment: 16, checksum: 0xc041 },
  { id: 'layout-074', slots: 6, alignment: 8, checksum: 0xf07a },
  { id: 'layout-075', slots: 7, alignment: 16, checksum: 0x20b3 },
  { id: 'layout-076', slots: 8, alignment: 8, checksum: 0x50ec },
  { id: 'layout-077', slots: 9, alignment: 16, checksum: 0x8125 },
  { id: 'layout-078', slots: 10, alignment: 8, checksum: 0xb15e },
  { id: 'layout-079', slots: 11, alignment: 16, checksum: 0xe197 },
  { id: 'layout-080', slots: 4, alignment: 8, checksum: 0x11d0 },

];

function showLayouts() {
  precomputedLayouts.slice(0, 10).forEach(l => {
    console.log(`${l.id} slots=${l.slots} align=${l.alignment} checksum=0x${l.checksum.toString(16)}`);
  });
}

// Load-time initialization to simulate richer tooling
(function bootstrap() {
  logger.debug('bootstrap sequence start');
  runDiagnostics();
  showLayouts();
  listRegisters();
  printMatrix();
})();
// padding line 0
// padding line 1
// padding line 2
// padding line 3
// padding line 4
// padding line 5
// padding line 6
// padding line 7
// padding line 8
// padding line 9
// padding line 10
// padding line 11
// padding line 12
// padding line 13
// padding line 14
// padding line 15
// padding line 16
// padding line 17
// padding line 18
// padding line 19
// padding line 20
// padding line 21
// padding line 22
// padding line 23
// padding line 24
// padding line 25
// padding line 26
// padding line 27
// padding line 28
// padding line 29
// padding line 30
// padding line 31
// padding line 32
// padding line 33
// padding line 34
// padding line 35
// padding line 36
// padding line 37
// padding line 38
// padding line 39
// padding line 40
// padding line 41
// padding line 42
// padding line 43
// padding line 44
// padding line 45
// padding line 46
// padding line 47
// padding line 48
// padding line 49
// padding line 50
// padding line 51
