#!/bin/bash

# Roundcube installation script
# Writeup: https://fearsoff.org/research/roundcube
# https://github.com/roundcube/roundcubemail/wiki/Installation

TARGET_VERSION=1.6.10 # latest vulnerable version

export DEBIAN_FRONTEND=noninteractive
export DEBCONF_NONINTERACTIVE_SEEN=true
export UCF_FORCE_CONFFOLD=1 
export DEBIAN_PRIORITY=critical

debconf-set-selections <<'EOF'
postfix postfix/main_mailer_type select Local only
postfix postfix/mailname string localhost
EOF

debconf-set-selections <<'EOF'
tzdata  tzdata/Areas select Etc
tzdata  tzdata/Zones/Etc select UTC
EOF
ln -sf /usr/share/zoneinfo/Etc/UTC /etc/localtime

echo "[1/11] Updating system and installing dependencies..."
apt-get update -qq
apt install -yq --no-install-recommends apache2 php php-mysql php-intl php-mbstring php-xml php-common php-cli php-curl php-zip php-gd php-imagick unzip mariadb-server dovecot-imapd dovecot-pop3d composer bsd-mailx wget curl

echo "[2/11] Downloading and extracting Roundcube ${TARGET_VERSION}..."
cd /tmp
wget -O roundcubemail.tar.gz wget https://github.com/roundcube/roundcubemail/releases/download/${TARGET_VERSION}/roundcubemail-${TARGET_VERSION}-complete.tar.gz
mkdir roundcubemail
tar --strip-components=1 -xzf roundcubemail.tar.gz -C roundcubemail
mv roundcubemail /var/www/html/roundcube

echo "[3/11] Installing PHP dependencies with Composer..."
cd /var/www/html/roundcube
mv composer.json-dist composer.json
composer install --no-dev
chown -R www-data:www-data /var/www/html/roundcube

echo "[4/11] Starting MariaDB …"
install -o mysql -g mysql -d /run/mysqld
install -o mysql -g mysql -d /var/log/mysql

# Initialise system tables once
if [ ! -d /var/lib/mysql/mysql ]; then
  mariadb-install-db --user=mysql --datadir=/var/lib/mysql --skip-test-db
fi

# Launch in background without explicit --log-error flag
mysqld_safe --datadir=/var/lib/mysql --socket=/run/mysqld/mysqld.sock &

# Wait until the socket is ready
echo -n "Waiting for MariaDB"; until mysqladmin ping --silent; do
  sleep 1; echo -n "."; 
done; echo " up!"

echo "[5/11] Creating Roundcube database and user..."
mysql -e "CREATE DATABASE roundcube CHARACTER SET utf8mb4 COLLATE utf8mb4_general_ci;"
mysql -e "CREATE USER roundcube@localhost IDENTIFIED BY 'fearsoff.org';"
mysql -e "GRANT ALL PRIVILEGES ON roundcube.* TO roundcube@localhost;"
mysql -e "FLUSH PRIVILEGES;"
mysql roundcube < /var/www/html/roundcube/SQL/mysql.initial.sql

echo "[6/11] Configuring Apache for roundcube.local..."
mkdir -p /var/www/html/roundcube/public_html
cat >/etc/apache2/sites-available/roundcube.conf <<EOF
<VirtualHost *:80>
    ServerName roundcube.local
    DocumentRoot /var/www/html/roundcube/public_html

    <Directory /var/www/html/roundcube/public_html>
        Options +FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
</VirtualHost>
EOF

echo "[✓] Adding roundcube.local to /etc/hosts..."
grep -q 'roundcube.local' /etc/hosts || echo "127.0.0.1 roundcube.local" >> /etc/hosts

ln -sfn /var/www/html/roundcube /var/www/html/roundcube/public_html

a2ensite roundcube.conf
a2enmod rewrite
a2dissite 000-default.conf

echo 'ServerName roundcube.local' > /etc/apache2/conf-available/servername.conf
a2enconf servername

service apache2 restart

echo "[7/11] Setting Roundcube configuration..."
cp /var/www/html/roundcube/config/config.inc.php.sample /var/www/html/roundcube/config/config.inc.php
sed -i "s#^\$config\['db_dsnw'\].*#\$config['db_dsnw'] = 'mysql://roundcube:fearsoff.org@localhost/roundcube';#" /var/www/html/roundcube/config/config.inc.php
echo "\$config['default_host'] = 'localhost';" >> /var/www/html/roundcube/config/config.inc.php
echo "\$config['smtp_server'] = 'localhost';" >> /var/www/html/roundcube/config/config.inc.php

echo "[8/11] Configuring Dovecot for system mailboxes..."
sed -i "s|^#mail_location =.*|mail_location = mbox:~/mail:INBOX=/var/mail/%u|" /etc/dovecot/conf.d/10-mail.conf
sed -i "s|^#disable_plaintext_auth =.*|disable_plaintext_auth = no|" /etc/dovecot/conf.d/10-auth.conf
sed -i "s|^auth_mechanisms =.*|auth_mechanisms = plain login|" /etc/dovecot/conf.d/10-auth.conf

echo "[9/11] Restarting Dovecot and Apache services..."
/etc/init.d/dovecot restart
service apache2 restart

echo "[10/11] Creating system test user 'roundcube'..."
useradd -m roundcube
echo "roundcube:fearsoff.org" | chpasswd

service postfix start
echo "Sending test email to roundcube..."
echo "This is a test email to confirm Roundcube can access system mailboxes." | mail -s "Test Mail" roundcube

echo "[11/11] Removing installer for security purposes..."
rm -rf /var/www/html/roundcube/installer

echo "[✓] Roundcube is installed and accessible at: http://roundcube.local/ or http://127.0.0.1:9876/"
echo "[✓] Test login: user 'roundcube', password 'fearsoff.org'"
