# CVE-2025-46408
### Improper Hostname Verification in EagleEyes Lite Android Application

## 1. Overview

<img src="images/eagleeyes-icon.png" align="left" width="150" hspace="10"/>
<br clear="left"/>

- **Name**: EagleEyes(Lite)  
- **Version**: 2.0.0  
- **Vendor**: AVTECH  
- **CWE**: [CWE-297: Improper Validation of Certificate with Host Mismatch](https://cwe.mitre.org/data/definitions/297.html)  
- **CVSS**: 9.8 CRITICAL  
- **Vector String**: CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:U/C:H/I:H/A:H    
## 2. Summary
EagleEyes Lite (version 2.0.0) disables hostname verification during HTTPS communication by using `SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER`.  
As a result, the application accepts TLS certificates regardless of their Common Name (CN) or Subject Alternative Name (SAN) values, which allows an attacker to impersonate the legitimate server with any valid or self-signed certificate.  
An adversary positioned on the same network can exploit this weakness to perform a MITM attack, thereby intercepting or altering sensitive communication between the application and the AVTECH backend services.

## 3. Details
When the device runs on Android versions below 8.0, meaning `SDK_API_26` is set to `false`, the method does not return `GetHttpsUrlResponse()`.  
Instead, it executes the vulnerable logic inside the `try` block.  
```java
public static String GetHttpsResponse(String str) {
    if (SDK_API_26) {
        return GetHttpsUrlResponse(str);
    }
    try {
        ...
        X509HostnameVerifier x509HostnameVerifier = SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER;
        SSLSocketFactory socketFactory = SSLSocketFactory.getSocketFactory();
        socketFactory.setHostnameVerifier(x509HostnameVerifier);
        ...
    }
    ...
}
```
Here, the use of `SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER` disables proper hostname verification, which means the TLS connection succeeds even if the server certificate’s CN (Common Name) or SAN (Subject Alternative Name) does not match the requested hostname.   
As a result, an attacker can impersonate the legitimate server with a forged certificate and launch MITM attacks to intercept or modify sensitive communication.

## 4. Proof of Concept (PoC)
By running the Frida hooking script [hook.js](hook.js), the value of `SDK_API_26` was forcibly set to `false` in order to emulate a lower Android version. This allowed us to monitor whether the `GetHttpsResponse()` method was invoked.  
As a result, we confirmed that `GetHttpsResponse()` was successfully called during execution.


![PoC](images/eagleeyes-poc.png)
For insufficient encryption of sensitive information in parameters, please refer to [CVE-2025-50110](https://github.com/shinyColumn/CVE-2025-50110).  
## 5. Recommendations
The application should remove the use of `SSLSocketFactory.ALLOW_ALL_HOSTNAME_VERIFIER` and enforce strict hostname verification for all HTTPS connections. The server’s Common Name (CN) or Subject Alternative Name (SAN) fields must be validated against the requested hostname, using `HttpsURLConnection.getDefaultHostnameVerifier()` or an equivalent strict verifier.   
Additionally, any legacy fallback logic that disables hostname checks for older Android versions should be removed or replaced with secure implementations to ensure consistent TLS validation.


## 6. References
- https://www.cve.org/CVERecord?id=CVE-2025-46408
- https://nvd.nist.gov/vuln/detail/CVE-2025-46408
- https://github.com/shinyColumn/CVE-2025-50110
- https://github.com/shinyColumn/CVE-2025-50944
