# CVE-2025-43300: iOS/macOS DNG Image Processing Memory Corruption

## Overview

CVE-2025-43300 is a critical memory corruption vulnerability in Apple's image processing framework that affects iOS 18.6.1 and macOS systems. The vulnerability exists in the JPEG Lossless decompression code within RawCamera.bundle, triggered by inconsistencies between TIFF metadata and JPEG stream parameters in DNG files.

## Vulnerability Details

### Root Cause
The vulnerability stems from a metadata/stream inconsistency in DNG (Digital Negative) files:

1. **SamplesPerPixel** metadata in TIFF headers indicates expected color components
2. **SOF3 component count** in JPEG Lossless streams specifies actual data components  
3. When these values differ, allocation and processing become misaligned

### Memory Corruption Flow
```
1. Parser reads SamplesPerPixel from TIFF metadata (e.g., 2 components)
2. System allocates buffer: width × height × 2 components
3. JPEG decoder reads SOF3 component count (e.g., 1 component)
4. Decoder writes data based on inconsistent assumptions
5. Buffer overflow occurs when more data is written than allocated
```

### Attack Vector
- **Zero-click vulnerability**: No user interaction required
- **Automatic processing**: Triggered by viewing, importing, or receiving DNG files
- **Cross-platform impact**: Affects both iOS and macOS systems
- **Memory corruption**: Can lead to application crashes or potential code execution

## Tools Overview

This repository contains two Python tools for analyzing and reproducing CVE-2025-43300:

### 1. DNG Vulnerability Analyzer (`dng_vulnerability_analyzer.py`)

**Purpose**: Analyzes DNG file structure to identify vulnerability conditions

**Key Features**:
- Parses TIFF/DNG file structure including IFDs and SubIFDs
- Locates SamplesPerPixel metadata tags
- Identifies JPEG Lossless SOF3 markers and component counts
- Detects metadata/stream inconsistencies
- Reports exact byte offsets for modification

**Usage**:
```bash
python3 dng_vulnerability_analyzer.py <dng_file>
```

**Output**:
- File structure analysis
- Metadata tag locations and values
- JPEG stream component information
- Vulnerability assessment
- Specific offsets for POC creation

### 2. Safe Hex Modifier (`hex_modifier.py`)

**Purpose**: Creates proof-of-concept files by safely modifying specific bytes

**Key Features**:
- Verifies expected byte values before modification
- Creates SHA256 hashes for file tracking
- Generates binary diff reports
- Implements safety checks to prevent accidental corruption
- Supports both manual and automated POC creation

**Usage**:
```bash
# Create POC using known vulnerable offsets
python3 hex_modifier.py create-poc <input.dng>

# Manual byte modification
python3 hex_modifier.py modify <input.dng> <offset> <old_byte> <new_byte> <output.dng>

# Generate diff report
python3 hex_modifier.py diff <original.dng> <modified.dng>
```

## POC Creation Process

### Step 1: Analyze Target File
```bash
python3 dng_vulnerability_analyzer.py IMGP0847.DNG
```

This identifies:
- SamplesPerPixel metadata location
- JPEG SOF3 component count locations
- Current values and suggested modifications

### Step 2: Create Vulnerable Sample
The POC requires two specific byte modifications:

1. **Increase SamplesPerPixel metadata** (typically from 1 to 2)
2. **Decrease SOF3 component count** (typically from 2 to 1)

```bash
python3 hex_modifier.py create-poc IMGP0847.DNG
```

### Step 3: Verify POC
```bash
python3 dng_vulnerability_analyzer.py vuln_poc_IMGP0847.dng
```

Confirms the inconsistency exists and vulnerability is triggered.

## Why the POC Works

### Technical Mechanism

1. **Allocation Phase**:
   - Image parser reads SamplesPerPixel = 2
   - Allocates buffer for 2 components worth of pixel data
   - Buffer size = image_width × image_height × 2

2. **Processing Phase**:
   - JPEG Lossless decoder encounters SOF3 with 1 component
   - Parsing logic becomes confused about actual vs expected data size
   - Attempts to write data based on inconsistent assumptions

3. **Memory Corruption**:
   - More data written to buffer than originally allocated
   - Buffer overflow corrupts adjacent memory regions
   - Results in application crash or potential code execution

### File Format Specifics

DNG files use TIFF container format with embedded JPEG Lossless streams:
- **TIFF metadata** describes image properties and color information
- **JPEG streams** contain compressed pixel data
- **Trust relationship** exists between metadata and stream content
- **Validation gap** allows inconsistent values to reach processing code

## Impact Assessment

### Severity: Critical
- **CVSS Score**: High (zero-click, memory corruption, wide impact)
- **Affected Systems**: iOS 18.6.1, macOS systems with similar image processing
- **Attack Vector**: Network/Local file transfer
- **User Interaction**: None required
- **Scope**: All devices processing DNG files

### Real-World Scenarios
- Email attachments with DNG files
- AirDrop transfers
- Cloud photo synchronization
- Web downloads of photographer samples
- Messaging app media sharing

## Mitigation and Patches

### Vendor Response
- **Fixed in iOS 18.6.2**: Apple implemented proper metadata validation
- **macOS updates**: Similar fixes applied to macOS image processing
- **Root cause**: Added consistency checks between metadata and stream parameters

### Recommended Mitigations
1. **Update systems**: Install iOS 18.6.2 or later
2. **Input validation**: Verify metadata consistency in custom parsers
3. **Bounds checking**: Implement strict buffer size validation
4. **Fuzzing**: Regular testing of file format parsers

## File Structure and Technical Details

### TIFF/DNG Structure
```
TIFF Header → IFD Chain → SubIFDs → Image Data
├── Metadata Tags (SamplesPerPixel, Compression, etc.)
├── JPEG Lossless Streams (SOF3 markers)
└── Pixel Data (compressed)
```

### Key Vulnerability Locations
- **SamplesPerPixel Tag**: TIFF tag 0x0115 in IFD structures
- **SOF3 Markers**: JPEG Lossless Start of Frame (0xFFC3)
- **Component Count**: Byte offset +9 from SOF3 marker
- **Critical Mismatch**: When tag value ≠ component count

## Testing Guidelines

### Safe Testing Environment
1. **Use isolated systems**: VMs or dedicated test devices
2. **Create snapshots**: Before testing POC files
3. **Monitor crashes**: Check system logs and crash reports
4. **Network isolation**: Prevent unintended file sharing

### Expected Behavior
- **Vulnerable systems**: Application crash, memory corruption errors
- **Patched systems**: Graceful error handling or correct processing
- **Log indicators**: Memory access violations, segmentation faults

## Technical References

### File Format Documentation
- **TIFF 6.0 Specification**: Tag structure and IFD format
- **DNG Specification**: Adobe Digital Negative format
- **JPEG Standard**: Lossless compression (ITU-T T.87)

### Security Research
- **Memory corruption techniques**: Buffer overflow fundamentals  
- **File format fuzzing**: Automated vulnerability discovery
- **Parser security**: Input validation best practices

---

**Disclaimer**: This research is provided for educational and defensive security purposes only. Users are responsible for compliance with applicable laws and ethical guidelines.

## NOTE

The crush takes about 1-3 mins to be triggered. have fun
And for the POC you can run open vuln_file.dng to run the file or just preview it and it be excuted