# HTTP/1 Desync Vulnerability Testing Tools

This repository contains defensive security tools for detecting HTTP/1 request smuggling (desync) vulnerabilities based on groundbreaking research by **James Kettle** of PortSwigger.

## Research Attribution

This work is directly inspired by and implements techniques from:

- **"HTTP/1 Must Die"** by James Kettle (PortSwigger Research)
  - Original Research: https://portswigger.net/research/http1-must-die
  - Researcher: James Kettle (@albinowax)
  - Company: PortSwigger Web Security

The attack patterns, techniques, and detection methods implemented in these tools are based on James Kettle's pioneering work that revealed fundamental flaws in HTTP/1.1 protocol implementation across major web infrastructure providers.

### Research Impact
James Kettle's research demonstrated vulnerabilities affecting:
- Cloudflare, T-Mobile, GitLab, Netlify, Akamai
- Earned over $350,000 in bug bounties
- Led to critical infrastructure fixes across the web

## Overview

HTTP/1.1's ambiguous request boundary definitions create what James Kettle termed "extreme ambiguity" about where requests begin and end. This leads to parser discrepancies between front-end and back-end servers, enabling sophisticated request smuggling attacks. These tools help identify such vulnerabilities for defensive security testing.

## Tools Included

### 1. Go Testing Tool (`http1-desync-test.go`)

A comprehensive HTTP/1 desync vulnerability tester that implements multiple attack patterns.

#### Features
- **V-H (Visible-Hidden) Desync**: Tests for visible-hidden parser discrepancies
- **0.CL (Zero Content-Length) Desync**: Exploits Content-Length: 0 handling differences
- **Expect Header Desync**: Uses Expect: 100-continue for request smuggling
- **Double-Desync**: Tests for Response Queue Poisoning vulnerabilities
- **CL.TE Desync**: Tests Content-Length vs Transfer-Encoding conflicts

#### Installation

```bash
# Clone the repository
git clone https://github.com/nconsolo/echteeteepee.git
cd echteeteepee

# Install dependencies
go mod download

# Build the tool
go build -o http1-desync-test http1-desync-test.go
```

#### Usage

```bash
# Test all attack patterns
./http1-desync-test -target https://example.com

# Test specific attack pattern
./http1-desync-test -target https://example.com -test vh

# Enable verbose output
./http1-desync-test -target https://example.com -verbose

# Set custom timeout
./http1-desync-test -target https://example.com -timeout 10s
```

#### Command-line Options
- `-target`: Target URL (required)
- `-test`: Test type: `all`, `vh`, `0cl`, `expect`, `double`, `clte` (default: `all`)
- `-timeout`: Request timeout (default: `5s`)
- `-verbose`: Enable verbose output to see raw requests/responses

### 2. Nuclei Template (`nuclei-templates/http1-desync.yaml`)

Automated scanning template for Nuclei to detect HTTP/1 desync vulnerabilities at scale.

#### Installation

```bash
# Install Nuclei (if not already installed)
go install -v github.com/projectdiscovery/nuclei/v3/cmd/nuclei@latest

# Copy template to Nuclei templates directory
cp nuclei-templates/http1-desync.yaml ~/.nuclei-templates/
```

#### Usage

```bash
# Scan single target
nuclei -u https://example.com -t nuclei-templates/http1-desync.yaml

# Scan multiple targets
nuclei -l targets.txt -t nuclei-templates/http1-desync.yaml

# Scan with rate limiting
nuclei -u https://example.com -t nuclei-templates/http1-desync.yaml -rl 10
```

## Attack Patterns Explained

*All attack patterns are based on James Kettle's research findings and proven exploit techniques.*

### V-H (Visible-Hidden) Desync
**Source**: James Kettle's "HTTP/1 Must Die" research  
Exploits discrepancies where the front-end proxy sees one request boundary while the back-end server sees another. The attack payload embeds a hidden request within what appears to be header data to the front-end, but is interpreted as a separate request by the back-end.

**Example from research**: Used to demonstrate parser differences in major CDN providers.

### 0.CL (Zero Content-Length) Desync  
**Source**: James Kettle's IIS and T-Mobile vulnerability findings  
Leverages different interpretations of `Content-Length: 0` headers between servers. Some servers ignore the body when CL=0, while others still process it, leading to request boundary confusion.

**Real-world impact**: Successfully exploited against IIS servers and T-Mobile infrastructure.

### Expect Header Desync
**Source**: James Kettle's T-Mobile and LastPass research  
Exploits the `Expect: 100-continue` mechanism where front-end and back-end handle the continuation differently. This creates opportunities to smuggle requests while the front-end waits for a 100-continue response.

**Research context**: Demonstrated against T-Mobile and LastPass authentication systems.

### Double-Desync (Response Queue Poisoning)
**Source**: James Kettle's advanced desync methodology  
Chains multiple desync attacks to poison the response queue, allowing attackers to control what responses other users receive. This represents one of the most sophisticated attacks in the research.

**Critical finding**: Enables complete site takeover by serving malicious responses to legitimate users.

### CL.TE / TE.CL Conflicts
**Source**: Core HTTP/1.1 ambiguity research by James Kettle  
Exploits conflicts between `Content-Length` and `Transfer-Encoding: chunked` headers. Different servers prioritize these headers differently, creating parser discrepancies that enable request smuggling.

**Foundation**: These conflicts represent the fundamental flaw James Kettle identified in HTTP/1.1 specification.

## Detection Indicators

Based on James Kettle's research methodology, the tools look for:

### Primary Indicators (from original research)
- **Unexpected 404/403/401 responses**: Indicates smuggled requests reaching unauthorized endpoints
- **Response containing admin/restricted paths**: Suggests successful request smuggling to protected areas
- **Multiple HTTP responses in single connection**: Sign of response queue poisoning
- **Parser error responses (400 Bad Request)**: May indicate successful desync confusion

### Advanced Detection Methods
- **Response timing anomalies**: Delays suggesting server-side processing confusion
- **Content-length mismatches**: Responses with incorrect body lengths
- **Header injection artifacts**: Evidence of smuggled headers in responses
- **Cache poisoning indicators**: Unexpected cached responses affecting other users

### Vulnerability Severity Assessment
Following James Kettle's impact classification:
- **Critical**: Response Queue Poisoning enabling user impersonation
- **High**: Request smuggling to admin endpoints
- **Medium**: Parser discrepancies without exploitation proof
- **Low**: Benign desync behaviors without security impact

## Responsible Use

These tools are designed for:
- **Defensive security testing** on authorized systems
- **Vulnerability assessment** with proper permissions
- **Security research** in controlled environments

**DO NOT** use these tools:
- Against systems you don't own or lack authorization to test
- For malicious purposes or unauthorized access
- In production environments without proper precautions

## Technical Implementation Details

### Connection Management
Following James Kettle's methodology, the tools maintain persistent HTTP/1.1 connections to properly test for desync conditions. Each test creates a single TCP connection and sends multiple requests to observe parsing differences.

### Request Crafting
Attack payloads are carefully crafted based on the exact patterns James Kettle used in his research, ensuring compatibility with the server behaviors he documented.

### Response Analysis
The detection logic implements the same indicators James Kettle used to identify successful desync attacks during his $350K+ bug bounty research.

## Vulnerability Impact & Remediation

Based on James Kettle's findings and recommendations:

### Impact Categories
1. **Complete Site Takeover**: Through response queue poisoning
2. **Credential Theft**: Via request smuggling to authentication endpoints  
3. **Cache Poisoning**: Serving malicious content to legitimate users
4. **Privilege Escalation**: Accessing admin endpoints through smuggled requests

### Remediation Strategies (per James Kettle)
1. **Primary Solution**: Migrate upstream connections to HTTP/2
2. **Intermediate**: Enable HTTP request normalization 
3. **Detection**: Deploy regular desync vulnerability scanning
4. **Monitoring**: Implement connection-level anomaly detection

### Testing Recommendations
- Test during maintenance windows to avoid disrupting services
- Use rate limiting when scanning production systems
- Coordinate with security teams before large-scale testing
- Document findings using James Kettle's vulnerability classification

## References & Credits

### Primary Research
- **"HTTP/1 Must Die"** - James Kettle (PortSwigger Research)  
  https://portswigger.net/research/http1-must-die
- **"HTTP Desync Attacks: Request Smuggling Reborn"** - James Kettle  
  https://portswigger.net/research/http-desync-attacks-request-smuggling-reborn
- **Black Hat USA 2019 Presentation** - James Kettle  
  "HTTP Desync Attacks: Smashing into the Cell Next Door"

### Additional Resources
- [HTTP Request Smuggling - PortSwigger Academy](https://portswigger.net/web-security/request-smuggling)
- [Burp Suite HTTP Request Smuggler Extension](https://portswigger.net/bappstore/aaaa60ef945341e8a450217a54a11646)
- [CWE-444: Inconsistent Interpretation of HTTP Requests](https://cwe.mitre.org/data/definitions/444.html)

### Tool Credits
- **James Kettle (@albinowax)**: Original research and attack methodology
- **PortSwigger**: Research publication and HTTP Request Smuggler tool
- **ProjectDiscovery**: Nuclei scanning framework

## License

MIT License - See LICENSE file for details

## Disclaimer

These tools are provided for educational and defensive security purposes only. Users are responsible for ensuring they have proper authorization before testing any systems. The authors assume no liability for misuse or damage caused by these tools.