# PDF417 ID Validation Vulnerability PoC

## Overview

This repository contains a Proof of Concept (PoC) demonstrating critical security vulnerabilities in how various ID verification systems process PDF417 barcodes on driver's licenses. The vulnerability affects multiple commercial ID scanners, including VeriScan, TokenWorks products, and certain state-run verification applications.

**IMPORTANT:** This code is provided for security research purposes only. See the [LICENSE.md](./LICENSE.md) file for allowed usage.

## Research Ethics Statement
This research was conducted following:
- Coordinated vulnerability disclosure principles
- 120+ day vendor notification period (exceeded standard 90 days)
- Synthetic data only (no real personal information)
- Educational/defensive security purposes
- ISO/IEC 29147 vulnerability disclosure standards
- Responsible disclosure through MITRE CVE Program and CISA VDP

## Related CVEs
This PoC demonstrates vulnerabilities assigned:
- **CVE-2025-31337** - IDScan.net validation flaw
- **CVE-2025-31336** - TokenWorks validation vulnerability

[//]: # (- **CVE-2025-XXXXX** - AAMVA systemic standard flaws &#40;pending assignment&#41;)

## Publication Status
- **Publication Date**: July 1, 2025
- **CVE Status**: Published (despite CISA case closure)
- **VINCE Case**: VU#396042 (closed by CISA as "not a vulnerability")
- **Disclosure Period**: 120+ days completed
- **MITRE CVE List**:
    - [CVE-2025-31337](https://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2025-31337)
    - [CVE-2025-31336](https://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2025-31336)

## Coordination History and Vendor Response Summary

### Timeline
- **March 22, 2025**: Initial CVE request to MITRE
- **March 25, 2025**: IDScan.net acknowledgment and initial dismissal
- **March 28, 2025**: TokenWorks acknowledgment
- **April 3, 2025**: CISA VINCE case VU#396042 opened
- **April 9-16, 2025**: Extensive technical discussions with vendors
- **June 18, 2025**: CISA case closure decision
- **July 1, 2025**: Public disclosure (post-coordination)

### Vendor Responses and Technical Disagreements

#### IDScan.net (Joshua Sheetz, CISO/VP Engineering)
**Initial Response**: *"After investigating this, it looks like you are just spoofing the barcode data. This is a common standard."*

**Technical Position**:
- Claimed their parser "just reads data" and isn't responsible for validation
- Argued that impossible values (524-year-old individuals) are acceptable in a "parser"
- Dismissed implementation-specific vulnerabilities as AAMVA standard limitations
- Contradicted their own marketing claims about "hundreds of algorithmic security checks"

**Marketing vs. Reality Gap**:
- VeriScan website promises "Fake ID Detection / ID Authentication"
- Claims to "catch fake IDs" and perform "security checks"
- Technical analysis revealed minimal validation beyond format compliance

#### TokenWorks Response
- Acknowledged the issue as a "DMV implementation problem"
- Downplayed vendor responsibility for cryptographic validation
- No remediation timeline provided

#### CISA Decision (June 18, 2025)
**Final Determination**: *"CISA has decided that this is not a vulnerability as it is an understood risk in the AAMVA DL Standard."*

**Technical Analysis of CISA's Position**:
This decision appears to conflate standard limitations with implementation-specific vulnerabilities. The research documented distinct technical issues:

1. **Multi-Layered Vulnerability Classification**:
    - **Layer 1**: AAMVA standard cryptographic deficiencies (acknowledged)
    - **Layer 2**: Implementation-specific validation failures (dismissed by CISA)
    - **Layer 3**: Marketing claims vs. actual security capabilities (not addressed)

2. **Implementation-Specific Issues Documented**:
    - **CWE-20 (Improper Input Validation)**: Accepting physiologically impossible values
    - **CWE-345 (Insufficient Verification of Data Authenticity)**: No cryptographic checks
    - **CWE-770 (Allocation of Resources Without Limits)**: No rate limiting
    - **CWE-841 (Improper Enforcement of Behavioral Workflow)**: No pattern detection

## Technical Evidence and Vendor Rebuttals

### Documented Vulnerabilities

#### 1. Temporal Validation Bypass
**Finding**: VeriScan accepts birth dates from medieval times (e.g., year 1500)
**Vendor Response**: *"Where do you draw the line, especially in just a parser"*
**Technical Counter**: Human physiological limits provide clear validation boundaries (max ~122 years)

#### 2. Pattern Detection Failure
**Finding**: Same license number with different names accepted repeatedly
**Vendor Response**: *"We would not know if the first or second scan was real"*
**Technical Counter**: Pattern detection flags suspicious behavior regardless of which is authentic

#### 3. Marketing Claims Discrepancy
**Finding**: Product marketed as providing "security checks" and "fake ID detection"
**Vendor Response**: *"Using only our parsing tool...you are limited to what feature sets you have"*
**Technical Counter**: Marketing materials explicitly claim validation capabilities

#### 4. Commercial License Implications
**Finding**: CDL (Commercial Driver's License) bypass poses public safety risks
**Vendor Response**: Not specifically addressed
**Technical Impact**: Hazmat/transport credential spoofing potential

### Proof of Concept Evidence
Documentation includes validated bypasses of:
- Missouri ShowMeID government application
- VeriScan enterprise validation
- Multiple consumer ID verification apps
- Commercial driver's license systems

## Research Methodology and Ethics

### Responsible Disclosure Approach
Despite vendor dismissals and CISA case closure, this research maintains responsible disclosure principles:

1. **Extended Coordination Period**: 120+ days (exceeding standard 90-day period)
2. **Synthetic Data Only**: All demonstrations use clearly marked research data
3. **Vendor Engagement**: Extensive technical discussions and proposed solutions
4. **Public Safety Focus**: Emphasis on defensive improvements rather than exploitation

### Technical Validation Standards
All findings were validated against:
- **NIST SP 800-63B** identity verification guidelines
- **NIST Cybersecurity Framework v1.1** security controls
- **CWE/SANS Top 25** vulnerability classifications
- **ISO/IEC 29147** disclosure standards

## Repository Structure

```shell
./
├── Cargo.lock
├── Cargo.toml                  # Rust project configuration
├── generated_pdf417.png        # Example output barcode
├── LICENSE.md                  # Security Research License
├── README.md                   # This file
├── scan_proof                  # Scanner validation evidence
│   ├── veriscan/              # VeriScan bypass documentation
│   ├── show_me_id/            # Missouri ShowMeID evidence
├── vince_coordination/         # VINCE case documentation
│   ├── VU396042_thread.pdf     # Complete VINCE discussion
├── scripts
│   └── barcode_reader.sh       # Utility script for reading barcodes
└── src
    ├── main.rs                 # Entry point
    ├── states                  # State-specific implementations
    │   ├── california_cdl.rs   # California Commercial DL
    │   ├── california.rs       # California standard DL
    │   ├── florida.rs          # Florida DL
    │   ├── georgia.rs          # Georgia DL
    │   ├── illinois.rs         # Illinois DL
    │   ├── mod.rs              # States module
    │   ├── new_jersey.rs       # New Jersey DL
    │   ├── south_carolina.rs   # South Carolina DL
    │   └── texas.rs            # Texas DL
    └── utils                   # Utility functions
        ├── decoding            # PDF417 decoding utilities
        │   ├── base64_to_str.rs
        │   ├── decode.rs
        │   ├── errors.rs
        │   ├── mod.rs
        │   └── tests.rs
        ├── encoding            # PDF417 encoding utilities
        │   ├── encode.rs
        │   └── mod.rs
        └── mod.rs
```

## Security Vulnerability

The fundamental vulnerability lies in how verification systems validate driver's license barcodes:

1. **Superficial Format Checking:** Scanners verify that barcodes follow the AAMVA standard format but fail to cryptographically validate the data's authenticity.

2. **No Database Cross-Reference:** Systems don't verify that information in the barcode matches records at issuing agencies.

3. **Header/Offset Manipulation:** By calculating technically correct offset values while using synthetic data, completely fabricated credentials can pass validation.

4. **Logical Validation Failure:** Impossible values (e.g., medieval birth dates) are accepted without question.

5. **Commercial License Risk:** Even heightened-security documents like Commercial Driver's Licenses can be spoofed using the same techniques.

## Technical Findings vs. Industry Response

### Core Technical Issues Identified

#### Implementation-Specific Vulnerabilities
1. **Physiological Validation Gaps**
    - Birth dates from year 1400-1500 accepted as valid
    - Ages exceeding 524 years pass validation
    - Future issue/expiration dates accepted

2. **Behavioral Analysis Deficiencies**
    - No detection of sequential modifications to same license number
    - Lack of rate limiting on scanning operations
    - No flagging of statistically impossible patterns

3. **Authentication Architecture Issues**
    - Marketing claims of "security checks" not technically implemented
    - Parser vs. validator artificial distinction
    - Missing cryptographic verification layers

#### Systemic AAMVA Standard Issues
1. **Cryptographic Deficiencies**
    - No digital signature requirements
    - Missing public key infrastructure
    - Lack of issuer authentication mechanisms

2. **Validation Protocol Gaps**
    - No mandatory range checking specifications
    - Missing cross-field consistency requirements
    - Inadequate temporal validation guidelines

### Industry Response Analysis

The coordination process revealed concerning patterns in how security issues are addressed:

#### Vendor Deflection Strategies
1. **Scope Limitation**: Claiming "parser vs. validator" distinctions to avoid responsibility
2. **Standard Blame**: Attributing all issues to AAMVA rather than implementation choices
3. **Marketing Disconnect**: Maintaining security claims while disclaiming validation responsibility

#### Regulatory Coordination Challenges
1. **CISA Position**: Dismissing implementation vulnerabilities as "understood risks"
2. **CVE Assignment**: Resistance to vendor-specific vulnerability classification
3. **Public Safety Impact**: Limited consideration of real-world security implications

## Verification

To verify the authenticity of this release:

1. Import my GPG key: `gpg --import KEYS.asc`
#### Regulatory Coordination Challenges
2. Verify archive: `gpg --verify id-validation-poc.tar.gz.asc id-validation-poc.tar.gz`

If verification succeeds, you'll see a message indicating a "Good signature" from my key.

## State Implementations

This PoC includes implementations for eight states, each demonstrating state-specific encoding patterns:

### California ([`california.rs`](./src/states/california.rs), [`california_cdl.rs`](./src/states/california_cdl.rs))
- ZC subfile format with unique document discriminator pattern
- CDL format includes specialized commercial driving endorsements
- Vulnerability allows changing birth dates without detection

### Georgia ([`georgia.rs`](./src/states/georgia.rs))
- ZG offset format with county information fields
- Duplicated address information in two subfiles
- Algorithm-generated license number and document discriminator

### Illinois ([`illinois.rs`](./src/states/illinois.rs))
- ZI offset format with supplemental data fields
- Special pattern-based document discriminator
- Redundant fields not properly validated for consistency

### New Jersey ([`new_jersey.rs`](./src/states/new_jersey.rs))
- ZN offset format with unique derived fields
- Prefix formatting for names
- Special numerical fields derived from other credentials

### South Carolina ([`south_carolina.rs`](./src/states/south_carolina.rs))
- No subfile designator (uses DLDCA rather than DLDAQ)
- Different field ordering from other states
- Security field patterns not properly validated

### Texas ([`texas.rs`](./src/states/texas.rs))
- ZT subfile format unique to Texas
- Complex document discriminator derived from multiple fields
- Inventory control number validation weakness

### Florida ([`florida.rs`](./src/states/florida.rs))
- ZF subfile format with unique offset calculations
- Special document discriminator generation pattern
- Date formatting vulnerabilities

> ### Note: Modifiable Value Reference
> Throughout the code implementations, certain values can be safely modified without breaking validation:
> - **'9' characters**: Fields using sequences of 9s (e.g., `999999999`) can be modified with arbitrary values
> - **'X' characters**: Fields using X placeholders (e.g., `XXXXX`) can be replaced with any characters
> - **Dates**: Any dates can be modified, even impossible ones (e.g., medieval birth dates, future issue/exp dates)
> - **Names**: Personal names can be completely fabricated
> - **Addresses**: Address data can be modified with any values
>
> These modifiable fields highlight the lack of data consistency validation in scanner systems.

## Key Findings

1. **Multiple Affected Products:** The vulnerability affects various ID verification systems, including commercial products widely used for age verification.

2. **Cryptographic Weakness:** None of the tested systems implement cryptographic validation of the data or digital signatures.

3. **License Duplication:** The same driver's license number can be used with different names without triggering alerts in verification systems.

4. **Barcode Dimension Variance:** Each state requires slightly different PDF417 barcode dimensions for optimal scanning (noted in test functions).

5. **Public Safety Implications:** The ability to create synthetic commercial driving credentials poses significant public safety risks.

6. **Coordination Process Gaps:** Industry and regulatory responses demonstrate systemic issues in vulnerability disclosure processes.

## Technical Implementation Examples

### Creating Synthetic Barcodes

The code provides builder patterns for each state implementation:

```rust
// Example demonstrating temporal validation bypass
use crate::states::california::CaliforniaLicense;

let license = CaliforniaLicense::builder()
    .expiration_date("12312030")    // Future expiration (accepted)
    .last_name("RESEARCH")
    .first_name("TEMPORAL")
    .middle_name("BYPASS")
    .issue_date("07012025")
    .birth_date("01011500")         // Medieval date (accepted by VeriScan)
    .sex("1")
    .eye_color("BLU")
    .height("070 IN")
    .address("123 VALIDATION ST")
    .city("TESTTOWN")
    .state("CA")
    .zip_code("900010000")
    .license_number("T3ST1234")     // Synthetic license number
    .country("USA")
    .weight("150")
    .hair_color("BRO")
    .sequence("99999")
    .issuer_identification_number("636014")  // Valid CA IIN
    .redundant_eye_color("BLU")
    .alternative_hair_color("BRO")
    .build()
    .expect("Failed to build license");

// Generate barcode that passes VeriScan validation
let barcode_data = license.to_barcode();
```

### Running Tests

Each state implementation includes test functions that demonstrate the vulnerability:

```bash
# Run tests for all states
cargo test

# Run specific vulnerability demonstrations
cargo test -- --nocapture test_temporal_validation_bypass
cargo test -- --nocapture test_commercial_license_spoofing
```

## Installation and Usage

### Prerequisites

1. Rust and Cargo installed on your system
2. Required dependencies (listed in [Cargo.toml](./Cargo.toml))

### Installation

```bash
# Clone the repository
git clone https://github.com/coleleavitt/pdf417-validation-poc.git
cd pdf417-validation-poc

# Build the project
cargo build --release

# Run the interactive demonstration
./target/release/pdf417-validation-poc 
```

### Scanning Generated Barcodes

Generated barcodes will be saved as PNG files that can be presented to verification systems for testing:

```bash
# Using the provided script to read a generated barcode (requires zbar tools)
./scripts/barcode_reader.sh generated_pdf417.png
```

## AAMVA Standard Reference

The implementations use standardized AAMVA codes for physical characteristics:

### Eye Color Codes
- BLK - Black
- BLU - Blue
- BRO/BRN - Brown
- GRY/GRA - Gray
- GRN - Green
- HAZ/HZL - Hazel
- MAR - Maroon
- MUL - Multicolored
- PNK - Pink
- DIC - Dichromatic
- UNK - Unknown

### Hair Color Codes
- BAL - Bald
- BLK - Black
- BLN/BLD - Blonde
- BRO/BRN - Brown
- GRY/GRA - Grey
- RED/AUB - Red/Auburn
- SDY - Sandy
- WHI - White
- UNK - Unknown

## References and Standards
- [AAMVA DL/ID Card Design Standard](https://www.aamva.org/dl-id-standards/)
- [NIST SP 800-63B Identity Verification Guidelines](https://pages.nist.gov/800-63-3/sp800-63b.html)
- [NIST Cybersecurity Framework v1.1](https://www.nist.gov/cyberframework)
- [CWE/SANS Top 25 Most Dangerous Software Errors](https://cwe.mitre.org/top25/)
- [ISO/IEC 29147 Vulnerability Disclosure](https://www.iso.org/standard/45170.html)
- [MITRE CVE Program](https://cve.mitre.org/)
- [CISA Vulnerability Disclosure Policy](https://www.cisa.gov/vulnerability-disclosure-policy)

## Limitations and Ethical Considerations

1. **Physical Security Features**: Generated barcodes lack physical security features of real IDs (holograms, RFID, etc.)
2. **Legal Compliance**: All demonstrations use synthetic data with clear "RESEARCH USE ONLY" watermarks
3. **Coordination Challenges**: Industry resistance to addressing implementation-specific vulnerabilities
4. **Regulatory Gaps**: Limited framework for addressing systemic validation issues

## Future Research Directions

### Technical Improvements Needed
1. **Cryptographic Validation**: Implementation of digital signature verification
2. **Behavioral Analytics**: Pattern detection for suspicious scanning behavior
3. **Database Integration**: Real-time cross-referencing with issuing authorities
4. **Multi-Factor Validation**: Combining physical and digital verification methods

### Policy and Coordination Improvements
1. **Vulnerability Classification**: Better frameworks for standard vs. implementation issues
2. **Industry Accountability**: Clearer responsibility models for security claims
3. **Public Safety Integration**: Enhanced consideration of real-world impact
4. **International Coordination**: Cross-border identity verification standards

## Responsible Disclosure and Public Interest

This research is published in the public interest despite coordination challenges. The decision to proceed with publication is based on:

1. **Extended Coordination Period**: 120+ days of vendor engagement
2. **Public Safety Implications**: Commercial license spoofing risks
3. **Technical Merit**: Documented implementation-specific vulnerabilities
4. **Educational Value**: Improving security awareness and defensive capabilities

### Vendor Engagement Continuation

Despite disagreements during coordination, I remain available to:
- Assist with technical implementation improvements
- Support AAMVA standard enhancement efforts
- Collaborate on defensive security measures
- Provide additional technical analysis as needed

## Contact and Coordination

For questions, additional research findings, or security concerns, please contact:
- **Email**: cole@unwrap.rs (PGP preferred)
- **GitHub**: [github.com/coleleavitt](https://github.com/coleleavitt)
- **LinkedIn**: [linkedin.com/in/coleleavitt](https://linkedin.com/in/coleleavitt)

For ongoing coordination with vendors or regulatory bodies:
- **MITRE**: cve@mitre.org
- **CISA**: vdp@cisa.dhs.gov
- **VINCE Case**: VU#396042 (closed, but reference available)

## License

This project is provided under a strict Security Research License. See [LICENSE.md](./LICENSE.md) for full details.

**Note**: This research was conducted with the goal of improving security across the identity verification ecosystem. While coordination efforts encountered challenges, the technical findings remain valid and require industry attention to protect public safety and national security interests.