# CVE-2024-52005: ANSI Escape Sequence Injection in Git

## Overview

This repository contains a Proof-of-Concept (PoC) demonstrating CVE-2024-52005, a vulnerability in Git that allows ANSI escape sequence injection through the sideband channel during Git operations.

**CVE-2024-52005** is a **High severity** (CVSS 7.5) vulnerability affecting Git versions prior to 2.48.1, 2.47.3, 2.46.5, 2.45.4, and 2.44.3.

## The Problem

This PoC was created to demonstrate that certain hardened container images incorrectly claim "not_affected" status in their VEX (Vulnerability Exploitability eXchange) documents, despite shipping vulnerable Git versions.

### Real-World Impact

The vulnerability allows attackers to:
- **Hide malicious commands** in Git output using invisible ANSI codes
- **Fake security scan results** in CI/CD pipelines
- **Social engineer users** with fake error messages and security alerts
- **Poison build logs** to make malicious code appear as passing builds
- **Manipulate terminal display** during critical operations

## Quick Start

### Option 1: Test with Docker (No Local Clone Required)

Test directly against DHI ArgoCD container using this GitHub repo:

```bash
docker run --rm -it dhi.io/argocd:3 sh -c '
  git clone https://github.com/andrewd-cg/cve-2024-52005-poc.git /tmp/poc &&
  cd /tmp/poc &&
  sh docker-test.sh
'
```

This command:
1. Clones the PoC from GitHub into the container
2. Runs the test script inside the container
3. Shows if ANSI escape sequences pass through

If you see `^[[31m` or `^[[32m` in the output, the vulnerability is confirmed.

### Option 2: Local Test (One Command)

Clone the repo and run the test locally:

```bash
git clone https://github.com/andrewd-cg/cve-2024-52005-poc.git
cd cve-2024-52005-poc
./test_ansi_injection.sh
```

This script will:
1. Create a Git repository with a malicious hook containing ANSI codes
2. Clone it using a vulnerable Git version
3. Show if ANSI escape sequences pass through

### Using the Full Exploitation Framework

```bash
# Create malicious repository with social engineering payload
./exploit_cve_2024_52005.py create social_engineering

# Test against vulnerable container
./exploit_cve_2024_52005.py test

# View results
cat -v /tmp/cve-2024-52005-output.log
```

## Repository Contents

- **[test_ansi_injection.sh](test_ansi_injection.sh)** - Simple standalone test script
- **[exploit_cve_2024_52005.py](exploit_cve_2024_52005.py)** - Complete exploitation framework with 8 attack payloads
- **[QUICKSTART.md](QUICKSTART.md)** - Quick start guide with examples
- **[TECHNICAL_DETAILS.md](TECHNICAL_DETAILS.md)** - Complete technical analysis and exploitation guide

## Attack Scenarios

### 1. Supply Chain Attack via ArgoCD
```
Attacker creates public GitHub repo → ArgoCD clones it automatically →
Git hook injects ANSI codes → Hides backdoor in deployment logs
```

### 2. CI/CD Pipeline Poisoning
```
Developer clones repo → Malicious hook runs → Injects fake "Build PASSED" →
Real failures hidden → Vulnerable code deployed to production
```

### 3. Security Tool Bypass
```
Security scanner clones repo → Hook injects ANSI codes →
Shows "0 vulnerabilities" → Real vulnerabilities hidden →
Malicious code passes review
```

## Exploitation Techniques

The framework includes 8 pre-built attack payloads:

1. **fake_success** - Display fake success messages
2. **hide_command** - Hide malicious commands in output
3. **cursor_manipulation** - Overwrite warnings with fake success
4. **social_engineering** - Trick users into running attacker's script
5. **fake_vulnerability_scan** - Hide real vulnerabilities
6. **progress_bar** - Create fake progress indicators
7. **urgent_warning** - Create false urgency
8. **log_poisoning** - Inject fake success into CI/CD logs

## Why This Matters

Many container images claim "not_affected" for CVE-2024-52005 with the justification:
```json
{
  "justification": "vulnerable_code_cannot_be_controlled_by_adversary"
}
```

**This is incorrect** for containers designed to clone untrusted repositories:
- ArgoCD clones third-party Git repositories
- CI/CD tools process external repositories
- Security scanners analyze untrusted code

The attacker **does** control the vulnerable code path through Git hooks and sideband messages.

## Affected Versions

Git versions affected by CVE-2024-52005:
- All versions < 2.48.1
- All versions < 2.47.3
- All versions < 2.46.5
- All versions < 2.45.4
- All versions < 2.44.3

## Mitigation

### For Git Users
Update to a patched version:
```bash
# Check your version
git --version

# Update via package manager
apt update && apt upgrade git  # Debian/Ubuntu
brew upgrade git               # macOS
```

### For Container Image Maintainers
1. **Update Git** in your images to patched versions
2. **Update VEX documents** to correctly reflect vulnerability status
3. **Test your images** using this PoC

### Temporary Workarounds
If you cannot update immediately:
1. Only clone repositories from trusted sources
2. Review Git output through sanitized logging
3. Run Git operations in sandboxed environments

## Testing Your Environment

```bash
# Test any container image
docker run --rm -v $(pwd)/ansi-test-repo:/repo:ro YOUR_IMAGE:TAG \
  sh -c 'git clone /repo /tmp/test 2>&1' | cat -v

# Look for ANSI codes like ^[[31m in the output
```

## References

- [CVE-2024-52005 on NVD](https://nvd.nist.gov/vuln/detail/CVE-2024-52005)
- [Git Security Advisory](https://github.com/git/git/security/advisories/)
- [OpenVEX Specification](https://github.com/openvex/spec)

## Responsible Disclosure

This PoC was created for:
- Security research and education
- Demonstrating incorrect vulnerability assessments
- Helping organizations properly evaluate their risk
- Encouraging accurate VEX documentation

**For authorized security testing only.** Do not use this against systems you don't own or have permission to test.

## License

MIT License - See LICENSE file for details

## Author

Created as part of container image security research, demonstrating the importance of accurate vulnerability disclosure and assessment.

## Contributing

Found an issue or have improvements? Pull requests welcome!

- Report incorrect VEX assessments you find
- Add new exploitation techniques
- Improve detection methods
- Enhance documentation
