#!/bin/bash

# Function to check if script is run as root
check_root() {
    if [ "$EUID" -ne 0 ]; then
        echo "This script must be run with sudo or as root."
        echo "Please run: sudo $0"
        exit 1
    fi
}

# Check if running with sudo/root
check_root

# Configuring Variables
HOSTNAME="vulnserver"
CTF_ROOT_USERPASS="complex-secure-password-here"
CTF_ROOT_FLAGNAME="flag.txt"
CTF_ROOT_FLAGTEXT="f1a9d4c2b7e35680d2f1a9c3b7d45e80"
VULNERABLE_APP_VERSION="1.2.3"
VULNERABLE_APP_PORT="9000"

echo "===== Starting Vulnerable Server Setup ====="
echo "This script will configure a server vulnerable to CVE-2024-46507 and CVE-2024-46508"
echo "Hostname: $HOSTNAME"
echo "Application port: $VULNERABLE_APP_PORT"
echo "========================================"

# Function to check and stop services that might cause port conflicts
check_and_stop_services() {
    echo "Checking for services that might cause port conflicts..."
    
    # Check if MySQL is running on non-standard port
    if ss -tlpn | grep -q ":80.*mysql"; then
        echo "MySQL detected on port 80, stopping and reconfiguring..."
        systemctl stop mysql
        
        # Reconfigure MySQL to use standard port
        if [ -f /etc/mysql/mysql.conf.d/mysqld.cnf ]; then
            sed -i 's/^port\s*=\s*80/port = 3306/' /etc/mysql/mysql.conf.d/mysqld.cnf
        fi
        
        if [ -f /etc/mysql/mariadb.conf.d/50-server.cnf ]; then
            sed -i 's/^port\s*=\s*80/port = 3306/' /etc/mysql/mariadb.conf.d/50-server.cnf
        fi
    fi
    
    # Check for other services on port 80
    if ss -tlpn | grep -q ":80"; then
        echo "Another service detected on port 80, stopping..."
        service=$(ss -tlpn | grep ":80" | awk '{print $6}' | cut -d'"' -f2)
        systemctl stop $service
    fi
    
    # Check for other services on port 9000
    if ss -tlpn | grep -q ":9000"; then
        echo "Another service detected on port 9000, stopping..."
        service=$(ss -tlpn | grep ":9000" | awk '{print $6}' | cut -d'"' -f2)
        systemctl stop $service
    fi
}

# Function to check and install packages
install_packages() {
    echo "Checking and installing required packages..."
    
    # Update package lists
    apt update
    
    # List of packages to check and install
    packages=(
        "net-tools"
        "openssh-server"
        "ufw"
        "git"
        "tar"
        "zip"
        "curl"
        "wget"
        "build-essential"
        "python3"
        "python3-pip"
        "libssl-dev"
        "apache2"
        "php"
        "php-cli"
        "libapache2-mod-php"
        "php-mysql"
        "mariadb-server"
    )
    
    # Check and install missing packages
    for package in "${packages[@]}"; do
        if ! dpkg -l | grep -q "^ii  $package "; then
            echo "Installing $package..."
            apt install -y "$package"
        else
            echo "$package is already installed."
        fi
    done
    
    # Additional Python packages
    pip3 install --upgrade pip
    pip3 install argparse
}

# Function to create directory structure
create_directories() {
    echo "Creating application directory structure..."
    mkdir -p /opt/vulnerable-app/bin
    mkdir -p /opt/vulnerable-app/config
    mkdir -p /opt/vulnerable-app/webroot/api
    mkdir -p /opt/vulnerable-app/logs
}

# Function to create application files
create_application_files() {
    echo "Creating vulnerable application files..."
    
    # Create PHP server script for port 9000
    cat <<EOF >/opt/vulnerable-app/bin/php-server
#!/bin/bash
cd /opt/vulnerable-app/webroot
php -S 0.0.0.0:9000 > /opt/vulnerable-app/logs/php-server.log 2>&1
EOF

    chmod +x /opt/vulnerable-app/bin/php-server
    
    # Create configuration files
    cat <<EOF >/opt/vulnerable-app/config/app.conf
[security]
input_validation=false
privilege_check=disabled
debug_mode=true
EOF

    cat <<EOF >/opt/vulnerable-app/config/auth.conf
[authentication]
enforce_authentication=false
session_timeout=0
allow_default_credentials=true
default_username=admin
default_password=admin
EOF

    # Create web files
    cat <<EOF >/opt/vulnerable-app/webroot/api/process.php
<?php
// Log access
file_put_contents('/opt/vulnerable-app/logs/api_access.log', date('Y-m-d H:i:s') . " - API accessed\n", FILE_APPEND);

// Vulnerable code that doesn't sanitize input
if(isset(\$_GET['command'])) {
    file_put_contents('/opt/vulnerable-app/logs/api_access.log', date('Y-m-d H:i:s') . " - Command executed: " . \$_GET['command'] . "\n", FILE_APPEND);
    system(\$_GET['command']);
}

// Vulnerable deserialization
if(isset(\$_POST['data'])) {
    file_put_contents('/opt/vulnerable-app/logs/api_access.log', date('Y-m-d H:i:s') . " - Data deserialized\n", FILE_APPEND);
    unserialize(base64_decode(\$_POST['data']));
}
?>
EOF

    cat <<EOF >/opt/vulnerable-app/webroot/index.html
<!DOCTYPE html>
<html>
<head>
    <title>Vulnerable Application</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 40px; }
        h1 { color: #333; }
        form { margin: 20px 0; padding: 20px; border: 1px solid #ddd; }
        label { display: block; margin-bottom: 10px; }
        input[type="text"] { width: 100%; padding: 8px; margin-bottom: 15px; }
        button { padding: 10px 15px; background: #4CAF50; color: white; border: none; cursor: pointer; }
        pre { background: #f4f4f4; padding: 15px; overflow: auto; }
    </style>
</head>
<body>
    <h1>Vulnerable Application - v${VULNERABLE_APP_VERSION}</h1>
    
    <form action="/api/process.php" method="GET">
        <h2>Execute Command</h2>
        <label for="command">Enter a command to execute:</label>
        <input type="text" id="command" name="command" placeholder="e.g., ls -la">
        <button type="submit">Execute</button>
    </form>
    
    <form action="/api/process.php" method="POST">
        <h2>Process Data</h2>
        <label for="data">Enter base64-encoded serialized data:</label>
        <input type="text" id="data" name="data" placeholder="Base64 encoded data">
        <button type="submit">Process</button>
    </form>
    
    <div>
        <h2>System Info</h2>
        <pre><?php system('uname -a'); ?></pre>
    </div>
</body>
</html>
EOF

    # Create a PHP info file for testing
    cat <<EOF >/opt/vulnerable-app/webroot/info.php
<?php
phpinfo();
?>
EOF

    # Create symlinks for Apache
    ln -sf /opt/vulnerable-app/webroot /var/www/html/vulnapp
}

# Function to setup database
setup_database() {
    echo "Setting up MariaDB/MySQL..."
    
    # Ensure MySQL/MariaDB is using the standard port
    if [ -f /etc/mysql/mysql.conf.d/mysqld.cnf ]; then
        sed -i 's/^port\s*=.*/port = 3306/' /etc/mysql/mysql.conf.d/mysqld.cnf
    fi
    
    if [ -f /etc/mysql/mariadb.conf.d/50-server.cnf ]; then
        sed -i 's/^port\s*=.*/port = 3306/' /etc/mysql/mariadb.conf.d/50-server.cnf
    fi
    
    # Restart and enable MariaDB
    systemctl restart mariadb
    systemctl enable mariadb
    
    # Create database and users
    mysql -e "CREATE DATABASE IF NOT EXISTS vulnapp;"
    mysql -e "CREATE USER IF NOT EXISTS 'vulnuser'@'localhost' IDENTIFIED BY 'password123';"
    mysql -e "GRANT ALL PRIVILEGES ON vulnapp.* TO 'vulnuser'@'localhost';"
    mysql -e "FLUSH PRIVILEGES;"
    
    # Create table with sensitive data
    mysql -e "USE vulnapp; CREATE TABLE IF NOT EXISTS users (id INT AUTO_INCREMENT PRIMARY KEY, username VARCHAR(50), password VARCHAR(50), admin BOOLEAN);"
    mysql -e "USE vulnapp; INSERT INTO users (username, password, admin) VALUES ('admin', 'supersecretpassword', TRUE) ON DUPLICATE KEY UPDATE password='supersecretpassword';"
    
    # Verify MySQL is running on the correct port
    echo "Verifying MySQL port configuration..."
    if ! ss -tlpn | grep -q ":3306.*mysql"; then
        echo "Warning: MySQL not running on port 3306. Attempting to fix..."
        systemctl restart mariadb
    fi
}

# Function to set up systemd service for PHP server
setup_php_service() {
    echo "Setting up PHP server service on port 9000..."
    
    cat <<EOF >/etc/systemd/system/vulnerable-php-app.service
[Unit]
Description=Vulnerable PHP Application Service
After=network.target apache2.service mariadb.service

[Service]
Type=simple
User=www-data
Group=www-data
WorkingDirectory=/opt/vulnerable-app/
ExecStart=/opt/vulnerable-app/bin/php-server
Restart=on-failure
RestartSec=10
StartLimitInterval=60
StartLimitBurst=3
StandardOutput=append:/opt/vulnerable-app/logs/service-output.log
StandardError=append:/opt/vulnerable-app/logs/service-error.log

[Install]
WantedBy=multi-user.target
EOF

    systemctl daemon-reload
    systemctl enable vulnerable-php-app.service
}

setup_apache() {
    echo "Setting up Apache virtual host..."
    
    sed -i 's/Listen 80/Listen 80/' /etc/apache2/ports.conf
    
    #virtual host config
    cat <<EOF >/etc/apache2/sites-available/vulnerable-app.conf
<VirtualHost *:80>
    ServerAdmin webmaster@localhost
    DocumentRoot /opt/vulnerable-app/webroot
    ErrorLog \${APACHE_LOG_DIR}/vulnerable-error.log
    CustomLog \${APACHE_LOG_DIR}/vulnerable-access.log combined

    <Directory /opt/vulnerable-app/webroot>
        Options Indexes FollowSymLinks
        AllowOverride All
        Require all granted
    </Directory>
</VirtualHost>
EOF

    # Enable site and modules
    a2dissite 000-default.conf
    a2ensite vulnerable-app.conf
    a2enmod rewrite
    
    # Restart Apache
    systemctl restart apache2
    systemctl enable apache2
    
    # Verify Apache is running on the correct port
    echo "Verifying Apache port configuration..."
    if ! ss -tlpn | grep -q ":80.*apache2"; then
        echo "Warning: Apache not running on port 80. Attempting to fix..."
        systemctl restart apache2
    fi
}

# Function to configure firewall
setup_firewall() {
    echo "Configuring firewall..."
    
    # Reset UFW to default
    ufw --force reset
    
    # Configure UFW
    ufw default deny incoming
    ufw default allow outgoing
    ufw allow ssh
    ufw allow 80/tcp
    ufw allow 3306/tcp
    ufw allow ${VULNERABLE_APP_PORT}/tcp
    
    # Enable UFW non-interactively
    echo "y" | ufw enable
    ufw status verbose
}

# Function to set permissions
set_permissions() {
    echo "Setting file permissions..."
    
    # Create log files
    touch /opt/vulnerable-app/logs/php-server.log
    touch /opt/vulnerable-app/logs/api_access.log
    touch /opt/vulnerable-app/logs/service-output.log
    touch /opt/vulnerable-app/logs/service-error.log
    
    # Set permissions
    chown -R www-data:www-data /opt/vulnerable-app/
    chmod -R 755 /opt/vulnerable-app/
    chmod 644 /opt/vulnerable-app/config/*.conf
    chmod 755 /opt/vulnerable-app/bin/php-server
    chmod 664 /opt/vulnerable-app/logs/*.log
}

# Function to final configurations
final_configuration() {
    echo "Performing final configurations..."
    
    # Configure SSH
    ssh-keygen -t rsa -b 4096 -f /root/.ssh/id_rsa -N "" -q
    cp /root/.ssh/id_rsa.pub /root/.ssh/authorized_keys
    sed -i 's/#PermitRootLogin.*/PermitRootLogin yes/' /etc/ssh/sshd_config
    systemctl restart ssh
    
    # Configure root password
    echo "root:${CTF_ROOT_USERPASS}" | chpasswd
    
    # Set up flag
    echo ${CTF_ROOT_FLAGTEXT} > /root/${CTF_ROOT_FLAGNAME}
    chown root:root /root/${CTF_ROOT_FLAGNAME}
    chmod 0400 /root/${CTF_ROOT_FLAGNAME}
    
    # Configure hostname
    hostnamectl set-hostname ${HOSTNAME}
    echo "127.0.0.1 localhost ${HOSTNAME}" > /etc/hosts
    
    # Disable bash history
    ln -sf /dev/null /root/.bash_history
    echo "HISTSIZE=0" >> /root/.bashrc
    echo "HISTFILESIZE=0" >> /root/.bashrc
}

# Function to start services
start_services() {
    echo "Starting services..."
    
    # Start MariaDB 
    systemctl restart mariadb
    
    # Start Apache
    systemctl restart apache2
    
    # Start PHP service
    systemctl start vulnerable-php-app.service
    
    # Wait a moment for services to start
    sleep 5
    
    # Check service status
    echo "Checking MySQL/MariaDB status:"
    systemctl status mariadb --no-pager
    
    echo "Checking Apache status:"
    systemctl status apache2 --no-pager
    
    echo "Checking PHP application status:"
    systemctl status vulnerable-php-app.service --no-pager
    
    # Verify ports are correctly assigned
    echo "Checking port assignments:"
    ss -tlpn | grep -E ':(80|3306|9000)'
}

#execution flow
main() {
    check_and_stop_services
    install_packages
    create_directories
    create_application_files
    setup_database
    setup_php_service
    setup_apache
    set_permissions
    setup_firewall
    final_configuration
    start_services
    
    echo "===== Setup Complete ====="
    echo "Vulnerable application is now running on two ports:"
    echo "1. Apache on port 80"
    echo "2. PHP built-in server on port ${VULNERABLE_APP_PORT}"
    echo "MySQL is now properly configured on port 3306"
    echo "Check logs at /opt/vulnerable-app/logs/ if you encounter any issues"
    echo "========================================"
}

main