#!/usr/bin/env python3
"""Simple launcher without multiprocessing - uses subprocess instead"""

import subprocess
import time
import os
import httpx
import asyncio
from pathlib import Path

async def main():
    task_id = "task-cve-2024-32964-ssrf"

    print("=" * 60)
    print("AAA Security Assessment Framework (Simple Launcher)")
    print("=" * 60)
    print(f"Task ID: {task_id}")
    print()

    # Start Green Agent in subprocess (redirect output to files to avoid pipe deadlock)
    print("[1/4] Starting Green Agent...")
    green_log = open("/tmp/green_agent.log", "w")
    green_proc = subprocess.Popen(
        ["python", "main.py", "green", "--port", "9001"],
        stdout=green_log,
        stderr=subprocess.STDOUT,
        env=os.environ.copy()
    )

    # Wait for Green Agent to be ready
    await asyncio.sleep(5)

    # Check if Green Agent is responding
    try:
        async with httpx.AsyncClient() as client:
            resp = await client.get("http://localhost:9001/.well-known/agent-card.json", timeout=5.0)
            if resp.status_code == 200:
                print("      Green Agent ready at http://localhost:9001")
            else:
                print(f"      Error: Green Agent returned status {resp.status_code}")
                green_proc.kill()
                return
    except Exception as e:
        print(f"      Error: Green Agent not responding: {e}")
        green_proc.kill()
        return

    # Start White Agent in subprocess (redirect output to files to avoid pipe deadlock)
    print("[2/4] Starting White Agent...")
    white_log = open("/tmp/white_agent.log", "w")
    white_proc = subprocess.Popen(
        ["python", "main.py", "white", "--port", "9002"],
        stdout=white_log,
        stderr=subprocess.STDOUT,
        env=os.environ.copy()
    )

    # Wait for White Agent to be ready
    await asyncio.sleep(5)

    # Check if White Agent is responding
    try:
        async with httpx.AsyncClient() as client:
            resp = await client.get("http://localhost:9002/.well-known/agent-card.json", timeout=5.0)
            if resp.status_code == 200:
                print("      White Agent ready at http://localhost:9002")
            else:
                print(f"      Error: White Agent returned status {resp.status_code}")
                green_proc.kill()
                white_proc.kill()
                return
    except Exception as e:
        print(f"      Error: White Agent not responding: {e}")
        green_proc.kill()
        white_proc.kill()
        return

    # Send task to Green Agent
    print("[3/4] Sending task to Green Agent...")
    task_message = f"""
<task_id>{task_id}</task_id>
<white_agent_url>http://localhost:9002/</white_agent_url>
"""

    try:
        from a2a.client import A2ACardResolver, A2AClient
        from a2a.types import Part, TextPart, MessageSendParams, Message, Role, SendMessageRequest
        import uuid

        async with httpx.AsyncClient(timeout=600.0) as http_client:
            # Get agent card
            resolver = A2ACardResolver(httpx_client=http_client, base_url="http://localhost:9001")
            card = await resolver.get_agent_card()

            # Create client
            client = A2AClient(httpx_client=http_client, agent_card=card)

            # Send message
            message_id = uuid.uuid4().hex
            params = MessageSendParams(
                message=Message(
                    role=Role.user,
                    parts=[Part(TextPart(text=task_message))],
                    message_id=message_id,
                )
            )
            request_id = uuid.uuid4().hex
            req = SendMessageRequest(id=request_id, params=params)

            print("      Waiting for evaluation to complete (this may take several minutes)...")
            response = await client.send_message(request=req)

            print()
            print("=" * 60)
            print("[4/4] EVALUATION RESULTS")
            print("=" * 60)

            # Extract response
            from a2a.types import SendMessageSuccessResponse
            from a2a.utils import get_text_parts

            res_root = response.root
            if isinstance(res_root, SendMessageSuccessResponse):
                res_result = res_root.result
                if isinstance(res_result, Message):
                    text_parts = get_text_parts(res_result.parts)
                    if text_parts:
                        print(text_parts[0])
                    else:
                        print("No text response from Green Agent")
                else:
                    print(f"Unexpected result type: {type(res_result)}")
            else:
                print(f"Unexpected response type: {type(res_root)}")

    except Exception as e:
        print(f"Error during evaluation: {e}")
        import traceback
        traceback.print_exc()
    finally:
        print()
        print("Terminating agents...")
        green_proc.kill()
        white_proc.kill()
        green_proc.wait()
        white_proc.wait()
        print("Agents terminated.")
        print("Evaluation complete.")

if __name__ == "__main__":
    asyncio.run(main())
