#!/bin/bash
# Made with ✨ Magic ©️ Nur Mukhammad Agus (https://github.com/madfxr), 2024. Free and Open Source Software (FOSS).
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
WHITE='\033[0;97m'
NC='\033[0m'

os_name=""
os_version=""
kernel_version=""
nftables_status=""
nft_list_tables="None"
nft_set_rbtree_status=""
affected_status=""
kernel_detected=false
nftables_active=false
cvss_score="${RED}7.8 High${NC}"
cve_id="${RED}https://nvd.nist.gov/vuln/detail/CVE-2024-26581${NC}"

echo -e "${YELLOW}
 ██████╗██╗   ██╗███████╗    ██████╗  ██████╗ ██████╗ ██╗  ██╗      ██████╗  ██████╗ ███████╗ █████╗  ██╗
██╔════╝██║   ██║██╔════╝    ╚════██╗██╔═████╗╚════██╗██║  ██║      ╚════██╗██╔════╝ ██╔════╝██╔══██╗███║
██║     ██║   ██║█████╗█████╗ █████╔╝██║██╔██║ █████╔╝███████║█████╗ █████╔╝███████╗ ███████╗╚█████╔╝╚██║
██║     ╚██╗ ██╔╝██╔══╝╚════╝██╔═══╝ ████╔╝██║██╔═══╝ ╚════██║╚════╝██╔═══╝ ██╔═══██╗╚════██║██╔══██╗ ██║
╚██████╗ ╚████╔╝ ███████╗    ███████╗╚██████╔╝███████╗     ██║      ███████╗╚██████╔╝███████║╚█████╔╝ ██║
 ╚═════╝  ╚═══╝  ╚══════╝    ╚══════╝ ╚═════╝ ╚══════╝     ╚═╝      ╚══════╝ ╚═════╝ ╚══════╝ ╚════╝  ╚═╝
                           ✦✦✦ VULNERABILITY CHECKER FOR BGN INTERNAL ✦✦✦

${NC}"

is_vulnerable() {
    local major=$1
    local minor=$2
    local patch=$3

    kernel_version=$(uname -r | cut -d'-' -f1)
    kernel_major=$(echo "$kernel_version" | cut -d'.' -f1)
    kernel_minor=$(echo "$kernel_version" | cut -d'.' -f2)
    kernel_patch=$(echo "$kernel_version" | cut -d'.' -f3)

    if [ "$kernel_major" -gt "$major" ]; then
        return 0
    elif [ "$kernel_major" -eq "$major" ]; then
        if [ "$kernel_minor" -gt "$minor" ]; then
            return 0
        elif [ "$kernel_minor" -eq "$minor" ]; then
            if [ "$kernel_patch" -ge "$patch" ]; then
                return 0
            fi
        fi
    fi
    return 1
}

detect_distro() {
    echo -e "${YELLOW}[DEBUG] Display OS Information...${NC}"
    if [ -f /etc/os-release ]; then
        . /etc/os-release
        os_name="${GREEN}$NAME${NC}"
        os_version="${GREEN}$VERSION${NC}"
    else
        os_name="${RED}OS Name Not Found${NC}"
        os_version="${RED}OS Version Not Found${NC}"
    fi
}

get_kernel_version() {
    echo -e "${YELLOW}[DEBUG] Display the Current Kernel Version...${NC}"
    kernel_version="${GREEN}$(uname -r)${NC}"
    if [ -n "$kernel_version" ]; then
        kernel_detected=true
        echo -e "${GREEN}[DEBUG] Output: $kernel_version${NC}"
    else
        echo -e "${RED}[ERROR] Kernel Version Not Detected${NC}"
    fi
}

check_nftables() {
    echo -e "${YELLOW}[DEBUG] Checking the Status of nftables...${NC}"
    if command -v nft >/dev/null 2>&1; then
        if systemctl is-active --quiet nftables; then
            nftables_status="${RED}Active${NC}"
            nftables_active=true
        else
            nftables_status="${GREEN}Inactive${NC}"
            nft_list_tables="${GREEN}None${NC}"
        fi

        if [ "$nftables_active" = true ]; then
            echo -e "${YELLOW}[DEBUG] Display the nftables List Tables...${NC}"
            nft_list_tables="${RED}$(nft list tables 2>/dev/null)${NC}"
            if [ -z "$nft_list_tables" ]; then
                nft_list_tables="${GREEN}None${NC}"
            fi
        fi
    else
        nftables_status="${RED}nft Command Not Found${NC}"
    fi
}

check_nft_set_rbtree() {
    echo -e "${YELLOW}[DEBUG] Checking the Kernel Configuration for nft_set_rbtree...${NC}"
    config_file="/boot/config-$(uname -r)"
    if [ -f "$config_file" ]; then
        if grep -i CONFIG_NFT_RBTREE "$config_file" >/dev/null 2>&1; then
            nft_set_rbtree_status="${RED}Set${NC}"
        else
            nft_set_rbtree_status="${GREEN}Not Set${NC}"
        fi
    else
        nft_set_rbtree_status="${RED}Kernel Configuration File Not Found${NC}"
    fi
}

check_vulnerability() {
    echo -e "${YELLOW}[DEBUG] Checking for Vulnerabilities Based on Kernel Version...${NC}"

    if (is_vulnerable 5 5 0 && ! is_vulnerable 5 10 210) || \
       (is_vulnerable 5 11 0 && ! is_vulnerable 5 15 149) || \
       (is_vulnerable 5 16 0 && ! is_vulnerable 6 1 78) || \
       (is_vulnerable 6 2 0 && ! is_vulnerable 6 6 17) || \
       (is_vulnerable 6 7 0 && ! is_vulnerable 6 7 5); then
        affected_status="${RED}Affected${NC}"
    elif (is_vulnerable 2 6 27 && ! is_vulnerable 5 4 269) || \
         (is_vulnerable 5 10 210) || \
         (is_vulnerable 5 15 149) || \
         (is_vulnerable 6 1 78) || \
         (is_vulnerable 6 6 17) || \
         (is_vulnerable 6 7 5 && ! is_vulnerable 6 12 0); then
        affected_status="${GREEN}Not Affected${NC}"
    else
        affected_status="${GREEN}Not Affected${NC}"
    fi
}

echo -e "${GREEN}[DEBUG] Started Checking for CVE-2024-26581...${NC}"
detect_distro
get_kernel_version
check_nftables
check_nft_set_rbtree
check_vulnerability
echo -e "${GREEN}[DEBUG] Checking Completed!${NC}"

echo -e "${GREEN}[SUMMARY] Display of Checking Results:${NC}"
if [ "$kernel_detected" = true ]; then
    echo -e "‣ ${WHITE}OS Name:${NC} ${os_name}"
    echo -e "‣ ${WHITE}OS Version:${NC} ${os_version}"
    echo -e "‣ ${WHITE}Kernel Version:${NC} ${kernel_version}"
    echo -e "‣ ${WHITE}nftables Status:${NC} ${nftables_status}"
    echo -e "‣ ${WHITE}nftables List Tables:${NC} ${nft_list_tables}"
    echo -e "‣ ${WHITE}nft_set_rbtree Status:${NC} ${nft_set_rbtree_status}"
    echo -e "‣ ${WHITE}Vulnerability Status:${NC} ${affected_status}"

    if [ "$affected_status" = "${RED}Affected${NC}" ]; then
        echo -e "‣ ${WHITE}CVSS Score:${NC} ${cvss_score}"
        echo -e "‣ ${WHITE}CVE ID:${NC} ${cve_id}"
    fi
else
    echo -e "${RED}[ERROR] Kernel Version Not Detected${NC}"
fi
