# CVE-2024-22243  

**Author: Sean Pesce**  

This project contains an example web application that demonstrates exploitable scenarios for
[CVE-2024-22243](https://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2024-22243),
a URL-parsing vulnerability in the Java [Spring Framework](https://spring.io/)
(official disclosure [here](https://spring.io/security/cve-2024-22243)).


# Vulnerability  

Affected versions of Spring parse the
["userinfo" segment](https://en.wikipedia.org/wiki/Uniform_Resource_Identifier#Syntax) of URLs in a
unique way, potentially resulting in the extraction of a host name segment that differs from many
other common libraries.

The abnormal behavior is due to the following regular expression ("regex") in the
[`UriComponentsBuilder`](https://github.com/spring-projects/spring-framework/blob/2e07f9ab33d882876f46912fcea08030b2593d49/spring-web/src/main/java/org/springframework/web/util/UriComponentsBuilder.java#L79)
class (introduced by
[this commit](https://github.com/spring-projects/spring-framework/commit/a4484bb767805ec9397302f1738d33123fb35dfb)
in 2014):

```java
private static final String USERINFO_PATTERN = "([^@\\[/?#]*)";
```

This regex does not permit the "left bracket" character (`[`) in the user info segment. However,
Spring appears to be an outlier with this behavior, so calling `getHost()` on a
[`UriComponents`](https://docs.spring.io/spring-framework/docs/current/javadoc-api/org/springframework/web/util/UriComponents.html)
object constructed using [`UriComponentsBuilder.fromUriString`](https://docs.spring.io/spring-framework/docs/current/javadoc-api/org/springframework/web/util/UriComponentsBuilder.html#fromUriString%28java.lang.String%29)
or
[`UriComponentsBuilder.fromHttpUrl`](https://docs.spring.io/spring-framework/docs/current/javadoc-api/org/springframework/web/util/UriComponentsBuilder.html#fromHttpUrl%28java.lang.String%29)
can result in unexpected behavior. The
[`RestTemplate`](https://docs.spring.io/spring-framework/docs/current/javadoc-api/org/springframework/web/client/RestTemplate.html),
[`RestClient`](https://docs.spring.io/spring-framework/docs/current/javadoc-api/org/springframework/web/client/RestClient.html),
and [`WebClient`](https://docs.spring.io/spring-framework/docs/current/javadoc-api/org/springframework/web/reactive/function/client/WebClient.html)
classes are also affected due to their internal use of `UriComponentsBuilder`; therefore,
implementations can be rendered vulnerable even without direct use of `UriComponentsBuilder`.

For specially-crafted inputs, Spring will return a host name value that differs from all of the
following:

 * Modern web browsers, including:
   * Chrome (and other Chromium-based browsers)
   * Firefox
   * Safari
 * [`java.net.URI`](https://docs.oracle.com/en/java/javase/17/docs/api/java.base/java/net/URI.html) (reportedly only for specific Java versions; other versions raise a `URISyntaxException`)
 * [`java.net.URL`](https://docs.oracle.com/en/java/javase/21/docs/api/java.base/java/net/URL.html)
 * `curl`
 * [`android.net.Uri`](https://developer.android.com/reference/android/net/Uri)
 * [`okhttp3.HttpUrl`](https://square.github.io/okhttp/3.x/okhttp/okhttp3/HttpUrl.html)
 * (Python 3) [`urllib.parse.urlparse`](https://docs.python.org/3/library/urllib.parse.html#urllib.parse.urlparse)

(Note that this list is non-exhaustive.)  

This behavior potentially renders Spring-based web applications vulnerable to
[open redirect](https://cwe.mitre.org/data/definitions/601.html) and
[server-side request forgery (SSRF)](https://cwe.mitre.org/data/definitions/918.html) if the
dependent implementation uses trusted host names for authorization or other security-relevant
mechanisms.  

## Examples  

The example web application contains two vulnerable endpoints.

The first endpoint, `/redirect`, shows how Spring's abnormal URL parsing can result in an open
redirect. It can be exploited using a URL such as the following:

```
https://127.0.0.1[@evil.com
```

The second endpoint, `/health-check`, demonstrates how a mismatch in URL parsing between Spring and
the Java standard library `URL` class can result in server-side request forgery (SSRF). It can be
exploited using a URL such as the following:

```
https://evil.com[@127.0.0.1
```


## Usage  

To build this project with Maven, simply run the following command (tested with OpenJDK 17):

```
mvn clean package
```

Then, start the web app with a command such as the following:

```
java -jar seanpesce-cve-2024-22243.jar 9999
```

The web app will be accessible at `http://127.0.0.1:9999/`.


## Docker  

To build the docker image, run the following command:

```
docker build -t seanpesce-cve-2024-22243:latest .
```

Then, start the web app with a command such as the following:

```
docker run -i -e PORT=9999 -p 9999:9999 seanpesce-cve-2024-22243:latest
```

The web app will be accessible at `http://127.0.0.1:9999/` on the Docker host.


## Semgrep  

This repository also contains [semgrep](https://semgrep.dev/) rules to assist in scanning for
potentially-vulnerable code paths. [`spring-cve-2024-22243_loose.yaml`](semgrep/spring-cve-2024-22243_loose.yaml)
performs naive scans for any use of the vulnerable APIs; as such, it will often return a large
number of false positives. [`spring-cve-2024-22243_strict.yaml`](semgrep/spring-cve-2024-22243_strict.yaml)
attempts to use stricter logic and taint analysis; however, this has not been
thoroughly tested and has high potential to miss some vulnerable implementations (especially when not using
Semgrep Pro, which is required for cross-file analysis).  


## Other Resources  

* [NIST database entry](https://nvd.nist.gov/vuln/detail/CVE-2024-22243)
* [Git commit that fixed the vulnerability](https://github.com/spring-projects/spring-framework/commit/7ec5c994c147f0e168149498b1c9d4a249d69e87)
* Vulnerable implementation demo: [Vulnerable OAuth flow with open redirect](https://github.com/threedr3am/learnjavabug/blob/master/spring/spring-uricomponentsbuilder/src/main/java/com/threedr3am/bug/spring/uricomponentsbuilder/controller/OAuthController.java) by [threedr3am](https://x.com/threedr3am1) of SecCoder Security Lab
* [CVE-2024-22259](https://spring.io/security/cve-2024-22259), a second URL-parsing vulnerability that was discovered in the fallout of CVE-2024-22243
* [CVE-2024-22262](https://spring.io/security/cve-2024-22262), a third URL-parsing vulnerability that was discovered in the fallout of the first two findings
* [PortSwigger URL Bypass Cheat Sheet](https://portswigger.net/web-security/ssrf/url-validation-bypass-cheat-sheet), which includes payloads for CVE-2024-22243
